//! MS ADPCM decoding functionality.
use crate::input::{DecoderResult, DecoderError};
use crate::io::byteio::*;

const ADAPT_TABLE: [i32; 16] = [
    230, 230, 230, 230, 307, 409, 512, 614,
    768, 614, 512, 409, 307, 230, 230, 230
];

#[derive(Default)]
pub struct Predictor {
    pub sample1:    i32,
    pub sample2:    i32,
    pub delta:      i32,
    pub coef1:      i32,
    pub coef2:      i32,
}

impl Predictor {
    pub fn new() -> Self { Self::default() }
    pub fn expand_nibble(&mut self, nibble: u8) -> i16 {
        let mul = if (nibble & 8) == 0 { i32::from(nibble) } else { i32::from(nibble) - 16 };
        let pred = self.calc_pred() + self.delta.wrapping_mul(mul);
        self.update(pred.max(-0x8000).min(0x7FFF));
        self.delta = (ADAPT_TABLE[nibble as usize].wrapping_mul(self.delta) >> 8).max(16);
        self.sample1 as i16
    }
    fn calc_pred(&self) -> i32 {
        self.sample1.wrapping_mul(self.coef1).wrapping_add(self.sample2.wrapping_mul(self.coef2)) >> 8
    }
    fn update(&mut self, new_samp: i32) {
        self.sample2 = self.sample1;
        self.sample1 = new_samp;
    }
}

///! Decodes input data into interleaved audio buffer by adding newly decoded samples at the end
pub fn msadpcm_decode(src: &[u8], channels: usize, block_len: usize, adapt_coeffs: &[[i32; 2]], dst: &mut Vec<i16>) -> DecoderResult<()> {
    let mut pred = [Predictor::default(), Predictor::default()];

    for blk in src.chunks(block_len) {
        let mut br = MemoryReader::new_read(blk);
        for p in pred[..channels].iter_mut() {
            let coef_idx                = br.read_byte()? as usize;
            if coef_idx >= adapt_coeffs.len() { return Err(DecoderError::InvalidData); }
            p.coef1 = adapt_coeffs[coef_idx][0];
            p.coef2 = adapt_coeffs[coef_idx][1];
        }
        for p in pred[..channels].iter_mut() {
            p.delta                     = i32::from(br.read_u16le()?);
        }
        for p in pred[..channels].iter_mut() {
            let samp                    = br.read_u16le()? as i16;
            p.sample1                   = i32::from(samp);
        }
        for p in pred[..channels].iter_mut() {
            let samp                    = br.read_u16le()? as i16;
            p.sample2                   = i32::from(samp);
        }
        for p in pred[..channels].iter() {
            dst.push(p.sample2 as i16);
        }
        for p in pred[..channels].iter() {
            dst.push(p.sample1 as i16);
        }
        if channels == 1 {
            while br.left() > 0 {
                let idx                 = br.read_byte()?;
                dst.push(pred[0].expand_nibble(idx >> 4));
                dst.push(pred[0].expand_nibble(idx & 0xF));
            }
        } else {
            while br.left() > 0 {
                let idx                 = br.read_byte()?;
                dst.push(pred[0].expand_nibble(idx >> 4));
                dst.push(pred[1].expand_nibble(idx & 0xF));
            }
        }
    }
    Ok(())
}
