use crate::input::*;

fn read_len(br: &mut dyn ByteIO, op: u8, mask: u8) -> DecoderResult<usize> {
    let mut len = usize::from(op & mask);
    if len == 0 {
        loop {
            let add = usize::from(br.read_byte()?);
            if add == 0 {
                len += 255;
            } else {
                len += add + usize::from(mask);
                break;
            }
        }
    }
    Ok(len)
}

pub fn lzo_unpack(br: &mut dyn ByteIO, csize: usize, dst: &mut Vec<u8>) -> DecoderResult<()> {
    let end = br.tell() + (csize as u64);

    let mut op = br.read_byte()?;
    if op > 17 {
        br.read_extend(dst, (op - 17).into())?;
        op = br.read_byte()?;
        if op < 16 {
            return Err(DecoderError::InvalidData);
        }
    }

    let mut last_literal = false;
    let mut len;
    let mut offset;
    while br.tell() < end {
        match op {
            0x40..=0xFF => {
                len = usize::from(op >> 5) - 1;
                offset = usize::from(br.read_byte()?) * 8 + usize::from((op >> 2) & 7) + 1;
            },
            0x20..=0x3F => {
                len = read_len(&mut *br, op, 0x1F)?;
                op = br.read_byte()?;
                offset = usize::from(br.read_byte()?) * 64 + usize::from(op >> 2) + 1;
            },
            0x10..=0x1F => {
                len = read_len(&mut *br, op, 0x7)?;
                offset = if (op & 8) != 0 { 1 << 15 } else { 1 << 14 };
                op = br.read_byte()?;
                offset += usize::from(br.read_byte()?) * 64 + usize::from(op >> 2);
                if offset == (1 << 14) {
                    if len != 1 {
                        return Err(DecoderError::InvalidData);
                    }
                    break; //xxx maybe treat this as the only proper EOF?
                }
            },
            _ if !last_literal => {
                len = read_len(br, op, 0xF)?;
                br.read_extend(dst, len + 3)?;
                op = br.read_byte()?;
                if op >= 0x10 {
                    continue;
                }
                len = 1;
                offset = (1 << 11) + usize::from(br.read_byte()?) * 4 + usize::from(op >> 2) + 1;
            },
            _ => {
                len = 0;
                offset = usize::from(br.read_byte()?) * 4 + usize::from(op >> 2) + 1;
            }
        }
        if offset > dst.len() {
            return Err(DecoderError::InvalidData);
        }
        let mut src_pos = dst.len() - offset;
        for _ in 0..(len + 2) {
            let c = dst[src_pos];
            dst.push(c);
            src_pos += 1;
        }
        len = usize::from(op & 3);
        last_literal = len != 0;
        if len > 0 {
            br.read_extend(dst, len)?;
        }
        op = br.read_byte()?;
    }
    if br.tell() == end {
        Ok(())
    } else {
        Err(DecoderError::InvalidData)
    }
}

