//! Input handlers for various formats

use crate::io::byteio::{ByteIOError, ByteIO};
use crate::io::bitreader::BitReaderError;
use crate::io::codebook::CodebookError;

mod detect;
pub mod metadata;
pub use metadata::*;
pub mod util;

pub use util::ReadVGAPal;

/// A list of common errors appearing during input data handling
#[derive(Clone,Debug,PartialEq)]
pub enum DecoderError {
    /// Provided format was not found in the list
    FormatNotFound(String),
    /// Input file not found
    InputNotFound(String),
    /// Auxiliary input file not found
    AuxInputNotFound(String),
    /// Invalid input data was provided
    InvalidData,
    /// Provided input turned out to be incomplete.
    ShortData,
    /// Feature is not implemented.
    NotImplemented,
    /// Some bug in decoder. It should not happen yet it might.
    Bug,
    /// Decoding is finished
    EOF,
}

impl std::fmt::Display for DecoderError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            DecoderError::FormatNotFound(ref fmt) => write!(f, "no plugin found to handle the input format '{fmt}'"),
            DecoderError::InputNotFound(ref name) => write!(f, "input name '{name}' cannot be opened"),
            DecoderError::AuxInputNotFound(ref name) => write!(f, "auxiliary input name '{name}' cannot be opened"),
            DecoderError::InvalidData => write!(f, "invalid input data"),
            DecoderError::ShortData => write!(f, "unexpected EOF"),
            DecoderError::NotImplemented => write!(f, "unsupported feature"),
            DecoderError::Bug => write!(f, "internal bug"),
            DecoderError::EOF => write!(f, "file end is reached"),
        }
    }
}

/// A specialised `Result` type for decoding operations.
pub type DecoderResult<T> = Result<T, DecoderError>;

impl From<ByteIOError> for DecoderError {
    fn from(_: ByteIOError) -> Self { DecoderError::ShortData }
}

impl From<BitReaderError> for DecoderError {
    fn from(_: BitReaderError) -> Self { DecoderError::ShortData }
}

impl From<CodebookError> for DecoderError {
    fn from(_: CodebookError) -> Self { DecoderError::InvalidData }
}

#[allow(unused_macros)]
#[cfg(debug_assertions)]
macro_rules! validate {
    ($a:expr) => { if !$a { println!("check failed at {}:{}", file!(), line!()); return Err(DecoderError::InvalidData); } };
}
#[cfg(not(debug_assertions))]
macro_rules! validate {
    ($a:expr) => { if !$a { return Err(DecoderError::InvalidData); } };
}

/// Common interface for input data.
pub trait InputSource {
    /// Reports the number of streams in the file
    fn get_num_streams(&self) -> usize;

    /// Reports specific stream information
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo;

    /// Decodes the next frame
    ///
    /// On success stream number and decoded frame data belonging to that stream are returned
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)>;
}

/// Registry entry for multimedia input plugin
pub struct InputPlugin {
    /// Short format name e.g. 'vmd'
    pub name:       &'static str,
    /// Long format name e.g. 'Coktel Vision VMD'
    pub long_name:  &'static str,
    /// Function used to attempt and create input plugin for the provided name
    pub open:       fn(name: &str) -> DecoderResult<Box<dyn InputSource>>,
}

mod decoders;
pub use decoders::INPUT_PLUGINS as MMEDIA_PLUGINS;

/// Common interface for archive extractor.
pub trait ArchiveSource {
    /// Returns name of the next file to be extracted
    fn get_file_name(&mut self) -> DecoderResult<String>;
    /// Extracts the next file into the provided destination
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()>;
    /// Tells the plugin not to attempt conversion of output
    fn set_no_convert(&mut self);
}

/// Registry entry for archive input plugin
pub struct ArchivePlugin {
    /// Short format name e.g. 'gob'
    pub name:       &'static str,
    /// Long format name e.g. 'Beam Software GOB'
    pub long_name:  &'static str,
    /// Function used to attempt and create input plugin for the provided name
    pub open:       fn(name: &str) -> DecoderResult<Box<dyn ArchiveSource>>,
}

mod archives;
pub use archives::ARCHIVE_PLUGINS as ARCHIVE_PLUGINS;

/// Detected resource type
pub enum InputType {
    Multimedia(Box<dyn InputSource>),
    Archive(Box<dyn ArchiveSource>),
}

/// Attempts to create an input source for the provided format and name
pub fn open_input(format: &str, name: &str, archives_only: bool) -> DecoderResult<InputType> {
    if !archives_only {
        for plugin in MMEDIA_PLUGINS.iter() {
            if plugin.name == format {
                return (plugin.open)(name).map(|ret| InputType::Multimedia(ret));
            }
        }
    }
    for plugin in ARCHIVE_PLUGINS.iter() {
        if plugin.name == format {
            return (plugin.open)(name).map(|ret| InputType::Archive(ret));
        }
    }
    Err(DecoderError::FormatNotFound(format.to_owned()))
}

/// Attempts to create an input source for the provided name
///
/// It may pick wrong input handler though so caveat user!
pub fn open_unknown_input(name: &str, archives_only: bool) -> DecoderResult<InputType> {
    let mut detected = None;
    let mut is_archive = false;
    if !archives_only {
        detected = detect::detect_format(name, detect::MMEDIA_DETECTORS);
    }
    if let Some((fmtname, score)) = detect::detect_format(name, detect::ARCHIVE_DETECTORS) {
        if let Some((_refname, ref_score)) = detected {
            if score > ref_score {
                detected = Some((fmtname, score));
                is_archive = true;
            }
        } else {
            detected = Some((fmtname, score));
            is_archive = true;
        }
    }
    if let Some((fmtname, _score)) = detected {
        println!(" detected input as '{fmtname}'");
        return open_input(fmtname, name, is_archive);
    }
    if !archives_only {
        for plugin in MMEDIA_PLUGINS.iter() {
            if let Ok(ret) = (plugin.open)(name) {
                println!(" detected input as '{}'", plugin.name);
                return Ok(InputType::Multimedia(ret));
            }
        }
    }
    for plugin in ARCHIVE_PLUGINS.iter() {
        if let Ok(ret) = (plugin.open)(name) {
            println!(" detected input as '{}'", plugin.name);
            return Ok(InputType::Archive(ret));
        }
    }
    Err(DecoderError::FormatNotFound("autodetect".to_owned()))
}

pub fn copy_data(src: &mut dyn ByteIO, dst: &mut dyn ByteIO, mut size: usize) -> DecoderResult<()> {
    let mut buf = [0; 2048];
    while size > 0 {
        let cur_size = buf.len().min(size);
        src.read_buf(&mut buf[..cur_size])?;
        dst.write_buf(&buf[..cur_size])?;
        size -= cur_size;
    }
    Ok(())
}
