use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct VPX1Decoder {
    fr:      FileReader<File>,
    width:   usize,
    height:  usize,
    fps:     u16,
    pal:     [u8; 768],
    map:     Vec<u8>,
    got_map: bool,
    vdata:   Vec<u8>,
    frame:   Vec<u8>,
    arate:   u16,
}

impl VPX1Decoder {
    fn recon_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut clrs = MemoryReader::new_read(&self.vdata);

        for (modes, strip) in self.map.chunks_exact(self.width / 16)
                .zip(self.frame.chunks_exact_mut(self.width * 8)) {
            for (x, &mode) in modes.iter().enumerate() {
                for &(x_off, mode) in [(x * 16, mode >> 4), (x * 16 + 8, mode & 0xF)].iter() {
                    Self::paint_block(&mut strip[x_off..], self.width, mode, &mut clrs)?;
                }
            }
        }

        let frm = self.frame.to_vec();
        let pal = self.pal;
        Ok((0, Frame::VideoPal(frm, pal)))
    }
    fn paint_block(dst: &mut [u8], stride: usize, mode: u8, clrs: &mut dyn ByteIO) -> DecoderResult<()> {
        match mode {
            0 => {},
            1 => {
                for rows in dst.chunks_mut(stride * 2) {
                    for x in (0..8).step_by(2) {
                        let clr = clrs.read_byte()?;
                        rows[x] = clr;
                        rows[x + 1] = clr;
                        rows[x + stride] = clr;
                        rows[x + stride + 1] = clr;
                    }
                }
            },
            2 => {
                for &yoff in [0u8, 2, 4, 6, 1, 3, 5, 7].iter() {
                    let cur_row = &mut dst[usize::from(yoff) * stride..];
                    for pair in cur_row.chunks_exact_mut(2).take(4) {
                        let clr = clrs.read_byte()?;
                        pair[0] = clr;
                        pair[1] = clr;
                    }
                }
            },
            4 => {
                let clr = clrs.read_byte()?;
                for row in dst.chunks_mut(stride) {
                    for el in row[..8].iter_mut() {
                        *el = clr;
                    }
                }
            },
            _ => return Err(DecoderError::InvalidData),
        }
        Ok(())
    }
}

impl InputSource for VPX1Decoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.arate), channels: 1, sample_type: AudioSample::U8
                 }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        loop {
            let tag = if let Ok(tag) = br.read_tag() { tag } else { return Err(DecoderError::EOF); };
            match &tag {
                b"COLS" => {
                    br.read_buf(&mut self.pal)?;
                },
                b"SOUN" => {
                    let asize = br.read_u32le()? as usize;
                    validate!(asize > 0);
                    let mut adata = vec![0; asize];
                    br.read_buf(&mut adata)?;
                    return Ok((1, Frame::AudioU8(adata)));
                },
                b"CODE" => {
                    br.read_buf(&mut self.map)?;
                    self.got_map = true;
                },
                b"VIDE" => {
                    validate!(self.got_map);
                    self.got_map = false;
                    let vsize = br.read_u32le()? as usize;
                    self.vdata.resize(vsize, 0);
                    br.read_buf(&mut self.vdata)?;
                    return self.recon_frame();
                },
                _ => {
                    println!("Unknown tag {:02X}{:02X}{:02X}{:02X} @ {:X}", tag[0], tag[1], tag[2], tag[3], br.tell() - 4);
                    return Err(DecoderError::NotImplemented);
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let mut hdr = [0; 120];
    fr.read_buf(&mut hdr)?;
    if &hdr[..56] != b"VPX1  video interflow packing exalter  video/audio codec" || &hdr[110..114] != b"HEAD" {
        return Err(DecoderError::InvalidData);
    }
    let width  = fr.read_u16le()? as usize;
    let height = fr.read_u16le()? as usize;
    let fps    = fr.read_u16le()?;
    let arate  = fr.read_u16le()?;
    validate!(width > 0 && width < 1024 && (width & 15) == 0);
    validate!(height > 0 && height < 1024 && (height & 15) == 0);

    Ok(Box::new(VPX1Decoder {
        fr,
        width, height, fps, arate,
        pal:     [0; 768],
        map:     vec![0; (width / 8) * (height / 8) / 2],
        got_map: false,
        vdata:   Vec::new(),
        frame:   vec![0; width * height],
    }))
}
