use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 640;
const HEIGHT: usize = 320;
const ABLK_LEN: usize = 1470;

#[derive(Clone,Copy,Default)]
struct Tile {
    mask:   u16,
    clr:    [u8; 2],
}

struct VDXDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    udata:      Vec<u8>,
    data:       Vec<u8>,
    win:        Vec<u8>,
    tile_w:     usize,
    tile_h:     usize,
    abuf:       Vec<u8>,
    audio:      bool,
    old_mode:   bool,
    tiles:      Vec<Tile>,
}

impl VDXDecoder {
    fn paint_block(slice: &mut [u8], x: usize, mut mask: usize, clr: [u8; 2]) {
        for line in slice.chunks_exact_mut(WIDTH) {
            for pix in line[x..][..4].iter_mut() {
                *pix = clr[(!mask >> 15) & 1];
                mask <<= 1;
            }
        }
    }
    fn decode_intra_old(&mut self) -> DecoderResult<()> {
        validate!(self.data.len() > 0x306);
        let (hdr, rest) = self.data.split_at(6);
        self.tile_w = usize::from(read_u16le(&hdr[0..])?);
        self.tile_h = usize::from(read_u16le(&hdr[2..])?);
        validate!(self.tile_w * 4 <= WIDTH && self.tile_h * 4 <= HEIGHT);
        let (pal, data) = rest.split_at(0x300);
        self.pal.copy_from_slice(pal);

        let part_size = self.tile_w * self.tile_h * 2;
        validate!(data.len() >= part_size * 2);
        let (flags, clrs) = data.split_at(part_size);

        self.tiles.resize(self.tile_w * self.tile_h, Tile::default());

        let mut iter = self.tiles.iter_mut().zip(flags.chunks_exact(2).zip(clrs.chunks_exact(2)));

        for slice in self.frame.chunks_exact_mut(WIDTH * 4).take(self.tile_h) {
            for x in (0..self.tile_w * 4).step_by(4) {
                let (tile, (flg, clr)) = iter.next().unwrap();
                tile.mask = read_u16le(flg).unwrap_or_default();
                tile.clr = [clr[0], clr[1]];
                Self::paint_block(slice, x, usize::from(tile.mask), tile.clr);
            }
        }
        Ok(())
    }
    fn decode_delta_old(&mut self) -> DecoderResult<()> {
        let mask_size = self.tile_w * self.tile_h / 4;
        validate!(self.data.len() >= mask_size + 2);
        let (_hdr, rest) = self.data.split_at(2);
        let (masks, upd) = rest.split_at(mask_size);

        let mut br = MemoryReader::new_read(upd);
        let mut masks = masks.chunks_exact(2);
        let mut mpos = 0;
        let mut mask = 0;
        let mut tiles = self.tiles.iter_mut();
        for slice in self.frame.chunks_exact_mut(WIDTH * 4).take(self.tile_h) {
            for x in (0..self.tile_w * 4).step_by(4) {
                if mpos == 0 {
                    let msk = masks.next().unwrap_or_default();
                    mask = read_u16le(msk).unwrap_or_default();
                    mpos = 16;
                }
                let mode = mask >> 14;
                mask <<= 2;
                mpos -= 2;
                let tile = tiles.next().unwrap();
                match mode {
                    1 => {
                        br.read_buf(&mut tile.clr)?;
                    },
                    2 => {
                        tile.mask = br.read_u16le()?;
                    },
                    3 => {
                        tile.mask = br.read_u16le()?;
                        br.read_buf(&mut tile.clr)?;
                    },
                    _ => {},
                }

                if mode != 0 {
                    Self::paint_block(slice, x, usize::from(tile.mask), tile.clr);
                }
            }
        }
        Ok(())
    }
    fn decode_intra(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);
        self.tile_w = usize::from(br.read_u16le()?);
        self.tile_h = usize::from(br.read_u16le()?);
        validate!(self.tile_w * 4 <= WIDTH && self.tile_h * 4 <= HEIGHT);
        let depth = br.read_u16le()?;
        validate!(depth == 8);
        br.read_buf(&mut self.pal)?;

        let mut clr = [0; 2];
        for slice in self.frame.chunks_exact_mut(WIDTH * 4).take(self.tile_h) {
            for x in (0..self.tile_w * 4).step_by(4) {
                br.read_buf(&mut clr)?;
                let mask = usize::from(br.read_u16le()?);
                Self::paint_block(slice, x, mask, clr);
            }
        }
        Ok(())
    }
    fn decode_delta(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let pal_upd_size = usize::from(br.read_u16le()?);
        if pal_upd_size > 0 {
            let mut mask = [0; 16];
            for el in mask.iter_mut() {
                *el = br.read_u16le()?;
            }
            for (&m, clrs) in mask.iter().zip(self.pal.chunks_exact_mut(3 * 16)) {
                let mut mask = m;
                for clr in clrs.chunks_exact_mut(3) {
                    if (mask & 0x8000) != 0 {
                        br.read_buf(clr)?;
                    }
                    mask <<= 1;
                }
            }
            validate!(br.tell() == (pal_upd_size as u64) + 2);
        }

        let mut skip = 0;
        let mut clr = [0; 2];
        let mut fill = 0;
        let mut fillclr = 0;
        let mut fillmode = false;
        for slice in self.frame.chunks_exact_mut(WIDTH * 4).take(self.tile_h) {
            let mut had_eol = false;
            for x in (0..self.tile_w * 4).step_by(4) {
                if skip > 0 {
                    skip -= 1;
                    continue;
                }
                if fill > 0 {
                    fill -= 1;
                    if fillmode {
                        fillclr = br.read_byte()?;
                    }
                    for line in slice.chunks_exact_mut(WIDTH) {
                        for pix in line[x..][..4].iter_mut() {
                            *pix = fillclr;
                        }
                    }
                    continue;
                }
                if br.left() == 0 {
                    return Ok(());
                }
                let op = br.read_byte()?;
                match op {
                    0x00..=0x5F => {
                        br.read_buf(&mut clr)?;
                        let mask = usize::from(TILE_MASK[usize::from(op)]);
                        Self::paint_block(slice, x, mask, clr);
                    },
                    0x60 => {
                        for line in slice.chunks_exact_mut(WIDTH) {
                            br.read_buf(&mut line[x..][..4])?;
                        }
                    },
                    0x61 => {
                        had_eol = true;
                        break;
                    },
                    0x62 => {},
                    0x63..=0x6B => {
                        skip = usize::from(op - 0x63);
                    },
                    0x6C..=0x75 => {
                        fill = usize::from(op - 0x6C);
                        fillclr = br.read_byte()?;
                        fillmode = false;
                        for line in slice.chunks_exact_mut(WIDTH) {
                            for pix in line[x..][..4].iter_mut() {
                                *pix = fillclr;
                            }
                        }
                    },
                    0x76..=0x7F => {
                        fill = usize::from(op - 0x76);
                        fillclr = br.read_byte()?;
                        fillmode = true;
                        for line in slice.chunks_exact_mut(WIDTH) {
                            for pix in line[x..][..4].iter_mut() {
                                *pix = fillclr;
                            }
                        }
                    },
                    _ => {
                        let mask = usize::from(op) | (usize::from(br.read_byte()?) << 8);
                        br.read_buf(&mut clr)?;
                        Self::paint_block(slice, x, mask, clr);
                    },
                }
            }
            if !had_eol {
                let op = br.read_byte()?;
                validate!(op == 0x61);
            }
        }
        Ok(())
    }
    fn lzss_unpack(&mut self, bits: u8, mask: u16) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);
        self.udata.clear();
        let mut bw = GrowableMemoryWriter::new_write(&mut self.udata);

        self.win.resize(1 << (16 - bits), 0);

        let mut wpos = 0;
        let wmask = self.win.len() - 1;
        let mut flags = 0;
        let mut fbits = 0;
        while br.left() > 0 {
            if fbits == 0 {
                flags = br.read_byte()?;
                fbits = 8;
            }
            if (flags & 1) != 0 {
                let b = br.read_byte()?;
                self.win[wpos] = b;
                wpos = (wpos + 1) & wmask;
                bw.write_byte(b)?;
            } else {
                let op = br.read_u16le()?;
                let len = (op & mask) as usize + 3;
                let offset = (op >> bits) as usize;
                if offset == 0 {
                    break;
                }

                let offset = wpos.wrapping_sub(offset) & wmask;
                for i in 0..len {
                    let b = self.win[(offset + i) & wmask];
                    self.win[(wpos + i) & wmask] = b;
                    bw.write_byte(b)?;
                }
                wpos = (wpos + len) & wmask;
            }
            flags >>= 1;
            fbits  -= 1;
        }

        std::mem::swap(&mut self.data, &mut self.udata);
        Ok(())
    }
    fn flip_buf(&self) -> Vec<u8> {
        let mut frame = Vec::with_capacity(WIDTH * HEIGHT);
        let (head, tail) = self.frame.split_at(WIDTH * self.tile_h * 4);
        for line in head.chunks_exact(WIDTH).rev()  {
            frame.extend_from_slice(line);
        }
        frame.extend_from_slice(tail);
        frame
    }
}

impl InputSource for VDXDecoder {
    fn get_num_streams(&self) -> usize { if !self.old_mode { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 15,
            }),
            1 if !self.old_mode => StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::U8,
                channels:    1,
            }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        loop {
            let br = &mut self.fr;
            if self.audio && (!matches!(br.peek_byte(), Ok(0x80)) || self.abuf.len() >= ABLK_LEN) {
                let mut audio = vec![0x80; ABLK_LEN];
                self.audio = false;
                if self.abuf.len() >= ABLK_LEN {
                    audio.copy_from_slice(&self.abuf[..ABLK_LEN]);
                    self.abuf.drain(..ABLK_LEN);
                }
                return Ok((1, Frame::AudioU8(audio)));
            }
            let ctype = br.read_byte().map_err(|_| DecoderError::EOF)?;
            if ctype == 0xFF { // end of VDX
                let next = br.read_byte().map_err(|_| DecoderError::EOF)?;
                match next {
                    0x67 => {
                        validate!(br.read_byte()? == 0x92);
                    },
                    0xFF => {
                        let tag = br.read_u16be().map_err(|_| DecoderError::EOF)?;
                        validate!(tag == 0x6792);
                    },
                    _ => return Err(DecoderError::EOF),
                }
                br.read_skip(6)?;
                self.pal = [0; 768];
                self.tile_w = 0;
                self.tile_h = 0;
                for el in self.frame.iter_mut() {
                    *el = 0;
                }
                self.abuf.clear();
                continue;
            }
            br.read_skip(1)?;
            let size = br.read_u32le()? as usize;
            validate!(size < 1048576);
            let lmask = br.read_byte()?;
            let lbits = br.read_byte()?;
            if !self.old_mode {
                validate!(lbits == 0 || (lbits <= 8 && u32::from(lmask) == (1 << lbits) - 1));
            } else {
                validate!(size < 0x10000);
            }
            self.data.resize(size, 0);
            br.read_buf(&mut self.data)?;
            if !self.old_mode && lbits > 0 {
                self.lzss_unpack(lbits, u16::from(lmask)).map_err(|_| DecoderError::InvalidData)?;
            }
            if !self.old_mode {
                match ctype {
                    0x00 => {
                        self.audio = true;
                        return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                    },
                    0x20 => {
                        self.decode_intra().map_err(|_| DecoderError::InvalidData)?;
                        self.audio = true;
                        return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                    },
                    0x25 => {
                        validate!(self.tile_w > 0);
                        self.decode_delta().map_err(|_| DecoderError::InvalidData)?;
                        self.audio = true;
                        return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                    },
                    0x80 => {
                        self.abuf.extend_from_slice(&self.data);
                        if self.audio && self.abuf.len() >= ABLK_LEN {
                            let mut audio = vec![0; ABLK_LEN];
                            audio.copy_from_slice(&self.abuf[..ABLK_LEN]);
                            self.abuf.drain(..ABLK_LEN);
                            self.audio = false;
                            return Ok((1, Frame::AudioU8(audio)));
                        }
                    },
                    _ => {},
                }
            } else {
                match ctype {
                    0x20 => {
                        self.decode_intra_old().map_err(|_| DecoderError::InvalidData)?;
                        return Ok((0, Frame::VideoPal(self.flip_buf(), self.pal)));
                    },
                    0x21 => {
                        validate!(self.tile_w > 0);
                        self.decode_delta_old().map_err(|_| DecoderError::InvalidData)?;
                        return Ok((0, Frame::VideoPal(self.flip_buf(), self.pal)));
                    },
                    _ => {},
                }
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_u16be()?;
    validate!(tag == 0x6792);
    fr.read_skip(6)?;
    let chunk_type = fr.peek_u16le()?;

    Ok(Box::new(VDXDecoder {
        fr,
        pal: [0; 768],
        frame: vec![0; WIDTH * HEIGHT],
        data: Vec::new(),
        udata: Vec::new(),
        win: Vec::new(),
        tile_w: 0,
        tile_h: 0,
        audio: false,
        abuf: Vec::new(),
        old_mode: (chunk_type >> 8) == 0x67,
        tiles: Vec::new(),
    }))
}

const TILE_MASK: [u16; 96] = [
    0xc800, 0xec80, 0xfec8, 0xffec, 0xfffe, 0x3100, 0x7310, 0xf731,
    0xff73, 0xfff7, 0x6c80, 0x36c8, 0x136c, 0x6310, 0xc631, 0x8c63,
    0xf000, 0xff00, 0xfff0, 0x1111, 0x3333, 0x7777, 0x6666, 0xcccc,
    0x0ff0, 0x00ff, 0xffcc, 0x0076, 0xff33, 0x0ee6, 0xccff, 0x6770,
    0x33ff, 0x6ee0, 0x4800, 0x2480, 0x1248, 0x0024, 0x0012, 0x2100,
    0x4210, 0x8421, 0x0042, 0x0084, 0xf888, 0x0044, 0x0032, 0x111f,
    0x22e0, 0x4c00, 0x888f, 0x4470, 0x2300, 0xf111, 0x0e22, 0x00c4,
    0xf33f, 0xfccf, 0xff99, 0x99ff, 0x4444, 0x2222, 0xccee, 0x7733,
    0x00f8, 0x00f1, 0x00bb, 0x0cdd, 0x0f0f, 0x0f88, 0x13f1, 0x19b3,
    0x1f80, 0x226f, 0x27ec, 0x3077, 0x3267, 0x37e4, 0x38e3, 0x3f90,
    0x44cf, 0x4cd9, 0x4c99, 0x5555, 0x603f, 0x6077, 0x6237, 0x64c9,
    0x64cd, 0x6cd9, 0x70ef, 0x0f00, 0x00f0, 0x0000, 0x4444, 0x2222
];
