use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct VDODecoder {
    fr:      FileReader<File>,
    width:   usize,
    height:  usize,
    pal:     [u8; 768],
    vdata:   Vec<u8>,
    frm:     Vec<u8>,
    version: u16,
}

impl InputSource for VDODecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        let x = br.read_u16le().map_err(|_| DecoderError::EOF)? as usize;
        let y = br.read_u16le()? as usize;
        let width = br.read_u16le()? as isize;
        let height = br.read_u16le()? as isize;
        let _smth = br.read_u32le()?;
        let size = br.read_u32le()? as usize;
        if width < 0 || height < 0 || size == 0 {
            return Ok((0, Frame::VideoPal(self.frm.clone(), self.pal)));
        }
        let width  = width as usize;
        let height = height as usize;
        validate!(x + width <= self.width && y + height <= self.height);
        validate!(size > height * 3);

        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;

        if self.version == 3 {
            for el in self.frm.iter_mut() { *el = 0; }
        }
        rle_unpack(&self.vdata, width, height, &mut self.frm[x + y * self.width..], self.width)
                .map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(self.frm.clone(), self.pal)))
    }
}

fn rle_unpack(src: &[u8], width: usize, height: usize, dst: &mut [u8], dstride: usize) -> DecoderResult<()> {
    let mut br = MemoryReader::new_read(src);
    for line in dst.chunks_exact_mut(dstride).take(height) {
        let data_size = br.read_u16le()?;
        validate!(br.left() >= i64::from(data_size));
        let data_end = br.tell() + u64::from(data_size);
        let mut dpos = 0;
        while dpos < width && br.tell() < data_end {
            let op = br.read_byte()?;
            let len = (op & 0x3F) as usize;
            validate!(dpos + len <= width);
            match op {
                0x00..=0x3F => {
                    let clr = br.read_byte()?;
                    for _ in 0..len {
                        line[dpos] = clr;
                        dpos += 1;
                    }
                },
                0x40..=0x7F => {
                    br.read_buf(&mut line[dpos..][..len])?;
                    dpos += len;
                },
                0x80..=0xBF => {
                    dpos += len;
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
        validate!(br.tell() == data_end);
    }
    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    let mut magic = [0; 8];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"Artform\0");
    let version = br.read_u16le()?;
    validate!(version <= 3);
    let width   = br.read_u16le()? as usize;
    let height  = br.read_u16le()? as usize;
    let _smth   = br.read_u16le()?;
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    Ok(Box::new(VDODecoder {
        fr: br,
        width, height,
        pal,
        vdata:   Vec::new(),
        frm:     vec![0; width * height],
        version,
    }))
}
