use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 256;
const HEIGHT: usize = 128;

struct UMDecoder {
    fr:         FileReader<File>,
}

impl InputSource for UMDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut frm = vec![0; WIDTH * HEIGHT];
        self.fr.peek_byte().map_err(|_| DecoderError::EOF)?;

        for plane in 0..4 {
            for dline in frm.chunks_exact_mut(WIDTH) {
                for quad in dline.chunks_exact_mut(4) {
                    quad[plane] = self.fr.read_byte()?;
                }
            }
        }

        let mut pal = [0; 768];
        self.fr.read_vga_pal(&mut pal)?;

        Ok((0, Frame::VideoPal(frm, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(file);

    Ok(Box::new(UMDecoder { fr }))
}
