use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;
use super::super::util::path::*;
use super::super::util::imaadpcm::*;

const DICT_SIZE: usize = 4096;
const INVALID_POS: usize = 65536;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
        }
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut Vec<u8>, pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= 256 {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        for _ in 0..tot_len {
            dst.push(0);
        }

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= 256 {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], lzw_bits: u8, dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() > 1);
        let mut br = BitReader::new(src, BitReaderMode::BE);

        self.dict_pos = 256;
        self.dict_lim = 1 << lzw_bits;

        let mut lastidx = br.read(lzw_bits).map_err(|_| DecoderError::InvalidData)? as usize;
        validate!(lastidx < 0x100);
        dst.push(lastidx as u8);
        loop {
            let ret         = br.read(lzw_bits);
            if ret.is_err() {
                return Err(DecoderError::InvalidData);
            }
            let idx = ret.unwrap() as usize;
            if idx == self.dict_lim - 1 {
                return Ok(());
            }
            validate!(idx <= self.dict_pos);
            let pos = dst.len();
            if idx != self.dict_pos {
                self.decode_idx(dst, pos, idx)?;
                self.add(lastidx, dst[pos]);
            } else {
                self.decode_idx(dst, pos, lastidx)?;
                let lastsym = dst[pos];
                dst.push(lastsym);
                self.add(lastidx, lastsym);
            }
            lastidx = idx;
        }
    }
}

struct FrameEntry {
    ftype:  u16,
    size:   u16,
    offset: u32,
}

struct MovieDecoder {
    fr:         FileReader<BufReader<File>>,
    frames:     Vec<FrameEntry>,
    frm_no:     usize,
    image:      Vec<u8>,
    vdata:      Vec<u8>,
    lzw:        LZWState,
    width:      usize,
    height:     usize,
    base_xoff:  i16,
    base_yoff:  i16,
    pal:        [u8; 768],
    data:       Vec<u8>,
    adata:      Vec<i16>,
    audio:      bool,
    adpcm:      IMAState,
}

impl MovieDecoder {
    fn decode_video(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);
        let _hdrsize = br.read_u16le()?;
        let width = usize::from(br.read_u16le()?);
        let stride = usize::from(br.read_u16le()?);
        let height = usize::from(br.read_u16le()?);
        let compr = br.read_u16le()?;
        validate!(stride >= width);
        validate!((1..=2048).contains(&width) && (1..=768).contains(&height));
        let xoff = br.read_u16le()? as i16;
        let yoff = br.read_u16le()? as i16;
        br.read_skip(6)?;

        let real_xoff = if xoff == self.base_xoff {
                0
            } else if xoff >= 0 {
                xoff as usize
            } else {
                let dx = usize::from(xoff.unsigned_abs());
                validate!(width < self.width);
                ((self.width - width) / 2).saturating_sub(dx)
            };
        let real_yoff = if yoff == self.base_yoff {
                0
            } else if yoff >= 0 {
                yoff as usize
            } else {
                let dy = usize::from(yoff.unsigned_abs());
                validate!(height < self.height);
                ((self.height - height) / 2).saturating_sub(dy)
            };

        self.vdata.clear();
        self.vdata.reserve(stride * height);
        match compr {
            0 | 1 | 2 => {
                loop {
                    let chunk = br.read_u16le()?;
                    if chunk == 0xE000 {
                        break;
                    }
                    let len = usize::from(chunk & 0x1FFF);
                    let mode = chunk >> 13;
                    validate!(len > 0);
                    if mode == 0 {
                        br.read_extend(&mut self.vdata, len)?;
                    } else {
                        let lzw_bits = match mode {
                            2 => 10,
                            3 => 11,
                            4 => 12,
                            _ => return Err(DecoderError::NotImplemented),
                        };
                        let len2 = usize::from(br.read_u16le()?);
                        validate!(len2 + 2 == len);
                        let start = br.tell() as usize;
                        br.read_skip(len2)?;
                        self.lzw.unpack(&self.data[start..][..len], lzw_bits, &mut self.vdata)?;
                    }
                }
            },
            0x80 | 0x82 => {
                for _ in 0..height {
                    br.read_u16le()?;
                }
                for _ in 0..height {
                    let mut pos = 0;
                    loop {
                        let op = br.read_byte()?;
                        match op {
                            0 => break,
                            1..=0x7F => {
                                let len = usize::from(op);
                                let clr = br.read_byte()?;
                                validate!(pos + len <= width);
                                for _ in 0..len {
                                    self.vdata.push(clr);
                                }
                                pos += len;
                            },
                            0xFF => {
                                let len = usize::from(br.read_byte()?);
                                validate!(pos + len <= width);
                                if len > 0 {
                                    br.read_extend(&mut self.vdata, len)?;
                                }
                                pos += len;
                            },
                            _ => {
                                let len = usize::from(!op);
                                validate!(pos + len <= width);
                                br.read_extend(&mut self.vdata, len)?;
                                pos += len;
                            },
                        }
                    }
                }
            },
            _ => return Err(DecoderError::NotImplemented),
        }

        let sstride = if self.vdata.len() > width * height { stride } else { width };

        for (dline, sline) in self.image.chunks_exact_mut(self.width).skip(real_yoff)
                .zip(self.vdata.chunks_exact(sstride).take(height).rev()) {
            match compr & 0x7F {
                0 => {
                    for (dst, &src) in dline.iter_mut().skip(real_xoff).zip(sline.iter()).take(width) {
                        *dst = src;
                    }
                },
                1 | 2 => {
                    for (dst, &src) in dline.iter_mut().skip(real_xoff).zip(sline.iter()).take(width) {
                        if src != 0 {
                            *dst = src;
                        }
                    }
                },
                _ => unreachable!(),
            }
        }

        Ok(())
    }
}

impl InputSource for MovieDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 8,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        const ABLK_LEN: usize = 22050 / 8;
        loop {
            if self.audio && self.adata.len() >= ABLK_LEN {
                self.audio = false;
                let mut ret = vec![0; ABLK_LEN];
                ret.copy_from_slice(&self.adata[..ABLK_LEN]);
                self.adata.drain(..ABLK_LEN);
                return Ok((1, Frame::AudioS16(ret)));
            }
            if self.frm_no >= self.frames.len() {
                return Err(DecoderError::EOF);
            }
            let frame = &self.frames[self.frm_no];
            self.frm_no += 1;
            self.data.resize(usize::from(frame.size), 0);
            self.fr.seek(SeekFrom::Start(frame.offset.into()))?;
            self.fr.read_buf(&mut self.data)?;
            match frame.ftype {
                0 | 1 => {
                    self.decode_video().map_err(|_| DecoderError::InvalidData)?;
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.image.clone(), self.pal)));
                },
                2 => {
                    for &b in self.data.iter() {
                        let samp = self.adpcm.expand_sample(b & 0xF);
                        self.adata.push(samp);
                        let samp = self.adpcm.expand_sample(b >> 4);
                        self.adata.push(samp);
                    }
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let mut bname = name.to_owned();
    match (bname.rfind('.'), bname.rfind(std::path::MAIN_SEPARATOR)) {
        (Some(epos), Some(spos)) => {
            if epos > spos {
                bname.truncate(epos);
            }
        },
        (Some(epos), None) => {
            bname.truncate(epos);
        },
        _ => {}
    }

    let toc_name = bname.clone() + ".mov_toc";
    let file = open_file_igncase(&toc_name).map_err(|_| DecoderError::AuxInputNotFound(toc_name))?;
    let mut br = FileReader::new_read(file);
    let mut frames = Vec::new();
    br.read_u16le()?;
    while let Ok(ftype) = br.read_u16le() {
        let size = br.read_u16le()?;
        let offset = br.read_u32le()?;
        frames.push(FrameEntry{ ftype, size, offset });
    }

    let frm_name = bname.clone() + ".mov_data";
    let file = open_file_igncase(&frm_name).map_err(|_| DecoderError::AuxInputNotFound(frm_name))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    if let Some(pos) = bname.rfind(std::path::MAIN_SEPARATOR) {
        bname.truncate(pos);
    } else {
        bname.truncate(0);
    }
    if !bname.is_empty() {
        bname.push(std::path::MAIN_SEPARATOR);
    }
    let pal_name = bname.clone() + "res0006";
    let pal_name2 = bname.clone() + ".." + std::path::MAIN_SEPARATOR_STR + "res0006";
    let mut file = open_file_igncase(&pal_name);
    if file.is_err() {
        file = open_file_igncase(&pal_name2);
    }
    let mut pal = std::array::from_fn(|i| (i / 3) as u8);
    if let Ok(fil) = file {
        let mut br = FileReader::new_read(fil);
        for clr in pal.chunks_exact_mut(3).skip(10) {
            if br.read_buf(clr).is_err() {
                break;
            }
            br.read_byte()?;
        }
    } else {
        println!("Palette file {pal_name} not found, defaulting to grayscale");
    }

    let mut width = 0;
    let mut base_xoff = 0;
    let mut height = 0;
    let mut base_yoff = 0;
    for frm in frames.iter() {
        if frm.ftype == 0 {
            validate!(frm.size > 20);
            fr.seek(SeekFrom::Start(frm.offset.into()))?;
            fr.read_u16le()?;
            width = usize::from(fr.read_u16le()?);
            fr.read_u16le()?;
            height = usize::from(fr.read_u16le()?);
            validate!((1..=640).contains(&width) && (1..=480).contains(&height));
            fr.read_u16le()?;
            base_xoff = fr.read_u16le()? as i16;
            base_yoff = fr.read_u16le()? as i16;
            break;
        }
    }
    validate!(width > 0);

    Ok(Box::new(MovieDecoder {
        fr,
        frames,
        frm_no: 0,
        data: Vec::new(),
        image: vec![0; width * height],
        lzw: LZWState::new(),
        vdata: Vec::new(),
        width, height, base_xoff, base_yoff, pal,
        audio: false,
        adata: Vec::new(),
        adpcm: IMAState::default(),
    }))
}
