use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::imaadpcm::*;
use super::super::*;

struct AudioDecoder {
    fr:         FileReader<BufReader<File>>,
    ima:        [IMAState; 2],
}

impl InputSource for AudioDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                channels:    2,
                sample_type: AudioSample::S16,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        const BLOCK_SIZE: usize = 0x200;

        let mut audio = Vec::with_capacity(BLOCK_SIZE * 4);
        let mut pair = [0; 2];
        for i in 0..BLOCK_SIZE {
            if self.fr.read_buf(&mut pair).is_err() {
                if i == 0 {
                    return Err(DecoderError::EOF);
                }
                break;
            }
            let samp0 = self.ima[0].expand_sample(pair[0] & 0xF);
            let samp2 = self.ima[0].expand_sample(pair[0] >> 4);
            let samp1 = self.ima[1].expand_sample(pair[1] & 0xF);
            let samp3 = self.ima[1].expand_sample(pair[1] >> 4);
            audio.push(samp0);
            audio.push(samp1);
            audio.push(samp2);
            audio.push(samp3);
        }

        Ok((0, Frame::AudioS16(audio)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    let mut ima = [IMAState::new(), IMAState::new()];
    ima[0].reset(0, 7);
    ima[1].reset(0, 7);
    Ok(Box::new(AudioDecoder {
        fr, ima,
    }))
}
