use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

struct EndianReader<'a> {
    br:     &'a mut dyn ByteIO,
    is_be:  bool,
}

impl<'a> EndianReader<'a> {
    fn read_byte(&mut self) -> ByteIOResult<u8> { self.br.read_byte() }
    fn read_tag(&mut self) -> ByteIOResult<[u8; 4]> { self.br.read_tag() }
    fn read_u16(&mut self) -> ByteIOResult<u16> {
        if self.is_be {
            self.br.read_u16be()
        } else {
            self.br.read_u16le()
        }
    }
    fn read_u32(&mut self) -> ByteIOResult<u32> {
        if self.is_be {
            self.br.read_u32be()
        } else {
            self.br.read_u32le()
        }
    }
    fn read_f32(&mut self) -> ByteIOResult<f32> {
        if self.is_be {
            self.br.read_f32be()
        } else {
            self.br.read_f32le()
        }
    }
    fn read_f64(&mut self) -> ByteIOResult<f64> {
        if self.is_be {
            self.br.read_f64be()
        } else {
            self.br.read_f64le()
        }
    }
    fn read_buf(&mut self, dst: &mut [u8]) -> ByteIOResult<()> { self.br.read_buf(dst) }
    fn read_skip(&mut self, size: usize) -> ByteIOResult<()> { self.br.read_skip(size) }
    fn tell(&mut self) -> u64 { self.br.tell() }
}

struct YUV2RGB {
    coeffs:     [f64; 9],
    remap:      [u16; 32768],
}

impl YUV2RGB {
    fn new() -> Self {
        let mut obj = Self {
                coeffs: [
                    1.007874,  1.413568,  0.0,
                    1.128063, -0.55146,  -0.347051,
                    1.007874,  0.0,       1.787011
                ],
                remap:  [0; 32768],
            };
        obj.init();
        obj
    }
    fn init(&mut self) {
        for (i, el) in self.remap.iter_mut().enumerate() {
            let y = (i >> 10) as f64;
            let u = ((i >> 5) & 0x1F) as f64 - 16.0;
            let v = (i & 0x1F) as f64 - 16.0;

            let r = y * self.coeffs[0] + u * self.coeffs[1] + v * self.coeffs[2];
            let g = y * self.coeffs[3] + u * self.coeffs[4] + v * self.coeffs[5];
            let b = y * self.coeffs[6] + u * self.coeffs[7] + v * self.coeffs[8];

            *el = ((r.max(0.0).min(31.0) as u16) << 10) |
                  ((g.max(0.0).min(31.0) as u16) << 5) |
                   (b.max(0.0).min(31.0) as u16);
        }
    }
}

struct PMMDecoder {
    fr:         FileReader<BufReader<File>>,
    is_be:      bool,
    width:      usize,
    height:     usize,
    fps:        u16,
    frame:      Vec<u16>,
    data:       Vec<u8>,
    end:        u64,
    adec:       AudioDecoder,
    aframe:     bool,
    yuv2rgb:    YUV2RGB,
    is_yuv:     bool,
    pal:        [u16; 256],
}

trait ReadLE {
    fn read_op(&mut self) -> BitReaderResult<u8>;
}

impl<'a> ReadLE for BitReader<'a> {
    fn read_op(&mut self) -> BitReaderResult<u8> {
        let mut val = 0;
        for _ in 0..3 {
            val <<= 1;
            if self.read_bool()? {
                val |= 1;
            }
        }
        Ok(val)
    }
}

trait ReadPixel {
    fn read_pixel(&mut self, tab: &[u16; 97]) -> DecoderResult<u16>;
    fn read_pixel4(&mut self, pixel: u16, index: usize) -> DecoderResult<u16>;
}

impl<T: ?Sized + ByteIO> ReadPixel for T {
    fn read_pixel(&mut self, tab: &[u16; 97]) -> DecoderResult<u16> {
        let b0 = self.read_byte()?;
        if (b0 & 0x80) == 0 {
            let idx = usize::from(b0);
            validate!(idx < tab.len());
            Ok(tab[idx])
        } else {
            let b1 = self.read_byte()?;
            Ok(u16::from(b0 & 0x7F) * 256 + u16::from(b1))
        }
    }
    fn read_pixel4(&mut self, pixel: u16, index: usize) -> DecoderResult<u16> {
        let b0 = self.read_byte()?;
        if (b0 & 0x80) == 0 {
            let idx = usize::from(b0);
            validate!(idx < PIX4_TABS[0].len());
            Ok((pixel & PIX4_TABS[index][idx]) | PIX8_TABS[index][idx])
        } else {
            let b1 = self.read_byte()?;
            Ok(u16::from(b0 & 0x7F) * 256 + u16::from(b1))
        }
    }
}

struct AudioDecoder {
    arate:    u16,
    audio:    Vec<i16>,
    apred:    [i16; 2],
    atype:    u8,
    channels: u8,
    ablk:     usize,
}

impl AudioDecoder {
    fn unpack_audio(&mut self, data: &[u8], is_be: bool) -> DecoderResult<()> {
        match self.atype {
            0 => {
                validate!(data.len() % (usize::from(self.channels) * 2) == 0);
                if is_be {
                    for samp in data.chunks_exact(2) {
                        self.audio.push(read_u16be(samp).unwrap_or_default() as i16);
                    }
                } else {
                    for samp in data.chunks_exact(2) {
                        self.audio.push(read_u16le(samp).unwrap_or_default() as i16);
                    }
                }
            },
            1 => {
                validate!(data.len() % usize::from(self.channels) == 0);
                for &b in data.iter() {
                    self.audio.push((u16::from(b) << 8) as i16);
                }
            },
            2 => {
                validate!(data.len() % usize::from(self.channels) == 0);
                for &b in data.iter() {
                    self.audio.push(PCM16_TABLE[usize::from(b)]);
                }
            },
            3 => {
                const ABLK_SIZE: usize = 16;
                validate!(data.len() % (usize::from(self.channels) * ABLK_SIZE) == 0);
                if self.channels == 1 {
                    for blk in data.chunks_exact(ABLK_SIZE) {
                        let shift = blk[0];
                        validate!((0..=15).contains(&shift));
                        for &b in blk[1..].iter() {
                            let delta = i16::from(b) << 12 >> 12 << shift;
                            self.apred[0] = self.apred[0].saturating_add(delta);
                            self.audio.push(self.apred[0]);
                            let delta = i16::from(b & 0xF0) << 8 >> 12 << shift;
                            self.apred[0] = self.apred[0].saturating_add(delta);
                            self.audio.push(self.apred[0]);
                        }
                    }
                } else {
                    for blks in data.chunks_exact(ABLK_SIZE * 2) {
                        let (blk0, blk1) = blks.split_at(ABLK_SIZE);
                        let shift0 = blk0[0];
                        let shift1 = blk1[0];
                        validate!((0..=15).contains(&shift0) && (0..=15).contains(&shift1));
                        for (&b0, &b1) in blk0[1..].iter().zip(blk1[1..].iter()) {
                            let delta = i16::from(b0) << 12 >> 12 << shift0;
                            self.apred[0] = self.apred[0].saturating_add(delta);
                            self.audio.push(self.apred[0]);
                            let delta = i16::from(b1) << 12 >> 12 << shift1;
                            self.apred[1] = self.apred[1].saturating_add(delta);
                            self.audio.push(self.apred[1]);
                            let delta = i16::from(b0 & 0xF0) << 8 >> 12 << shift0;
                            self.apred[0] = self.apred[0].saturating_add(delta);
                            self.audio.push(self.apred[0]);
                            let delta = i16::from(b1 & 0xF0) << 8 >> 12 << shift1;
                            self.apred[1] = self.apred[1].saturating_add(delta);
                            self.audio.push(self.apred[1]);
                        }
                    }
                }
            },
            _ => return Err(DecoderError::NotImplemented),
        }
        Ok(())
    }
}

struct DecState<'a> {
    pixel:  u16,
    mask:   u16,
    index:  usize,
    ops:    BitReader<'a>,
    clr:    MemoryReader<'a>,
}

impl<'a> DecState<'a> {
    fn read_pixel_update(&mut self) -> DecoderResult<u16> {
        self.pixel = self.clr.read_pixel4(self.pixel, self.index)?;
        Ok(self.pixel)
    }
}

impl PMMDecoder {
    fn do_block2x2(dst: &mut [u16], stride: usize, dec: &mut DecState) -> DecoderResult<()> {
        if !dec.ops.read_bool()? {
            dec.read_pixel_update()?;
            if dec.mask == 0 {
                for line in dst.chunks_mut(stride).take(2) {
                    for el in line[..2].iter_mut() {
                        *el = dec.pixel;
                    }
                }
            } else {
                for line in dst.chunks_mut(stride).take(2) {
                    for el in line[..2].iter_mut() {
                        *el = (*el & dec.mask) | dec.pixel;
                    }
                }
            }
        } else if dec.mask == 0 {
            dst[0]          = dec.read_pixel_update()?;
            dst[1]          = dec.read_pixel_update()?;
            dst[stride]     = dec.read_pixel_update()?;
            dst[stride + 1] = dec.read_pixel_update()?;
        } else {
            dst[0]          = (dst[0]          & dec.mask) | dec.read_pixel_update()?;
            dst[1]          = (dst[1]          & dec.mask) | dec.read_pixel_update()?;
            dst[stride]     = (dst[stride]     & dec.mask) | dec.read_pixel_update()?;
            dst[stride + 1] = (dst[stride + 1] & dec.mask) | dec.read_pixel_update()?;
        }
        Ok(())
    }
    fn do_block4x4(dst: &mut [u16], stride: usize, dec: &mut DecState) -> DecoderResult<()> {
        if !dec.ops.read_bool()? {
            dec.read_pixel_update()?;
            if dec.mask == 0 {
                for line in dst.chunks_mut(stride).take(4) {
                    for el in line[..4].iter_mut() {
                        *el = dec.pixel;
                    }
                }
            } else {
                for line in dst.chunks_mut(stride).take(4) {
                    for el in line[..4].iter_mut() {
                        *el = (*el & dec.mask) | dec.pixel;
                    }
                }
            }
        } else {
            Self::do_block2x2(dst, stride, dec)?;
            Self::do_block2x2(&mut dst[2..], stride, dec)?;
            Self::do_block2x2(&mut dst[stride * 2..], stride, dec)?;
            Self::do_block2x2(&mut dst[stride * 2 + 2..], stride, dec)?;
        }
        Ok(())
    }
    fn unpack_video(&mut self) -> DecoderResult<()> {
        const COMPONENT_MASK: [u16; 8] = [0x7FFF, 0x7FE0, 0x7C1F, 0x7C00, 0x03FF, 0x03E0, 0x001F, 0x0000 ];

        validate!(self.data.len() > 4);
        let code_off = usize::from(if self.is_be { read_u16be(&self.data[0..])? } else { read_u16le(&self.data[0..])? });
        let pix_off = usize::from(if self.is_be { read_u16be(&self.data[2..])? } else { read_u16le(&self.data[2..])? });
        validate!(code_off >= 4 && code_off < self.data.len());
        validate!(pix_off <= self.data.len());

        if self.is_be {
            validate!(pix_off > code_off && (pix_off - code_off) % 4 == 0);
            for quad in self.data[code_off..pix_off].chunks_exact_mut(4) {
                quad.swap(0, 3);
                quad.swap(1, 2);
            }
        }

        let ops = BitReader::new(&self.data[code_off..], BitReaderMode::LE);
        let clr = MemoryReader::new_read(&self.data[pix_off..]);
        let mut dec = DecState {
                pixel:  0,
                mask:   0,
                index:  0,
                ops, clr,
            };
        for strip in self.frame.chunks_exact_mut(self.width * 8) {
            for x in (0..self.width).step_by(8) {
                let op = usize::from(dec.ops.read_op()?);
                if op == 0 {
                    continue;
                }

                dec.pixel = 0;
                dec.index = op;
                dec.mask = COMPONENT_MASK[op];
                if !dec.ops.read_bool()? {
                    dec.pixel = dec.clr.read_pixel(PIX8_TABS[dec.index])?;
                    if dec.mask == 0 {
                        for line in strip.chunks_exact_mut(self.width) {
                            for el in line[x..][..8].iter_mut() {
                                *el = dec.pixel;
                            }
                        }
                    } else {
                        for line in strip.chunks_exact_mut(self.width) {
                            for el in line[x..][..8].iter_mut() {
                                *el = (*el & dec.mask) | dec.pixel;
                            }
                        }
                    }
                } else {
                    Self::do_block4x4(&mut strip[x..], self.width, &mut dec)?;
                    Self::do_block4x4(&mut strip[x + 4..], self.width, &mut dec)?;
                    Self::do_block4x4(&mut strip[x + 4 * self.width..], self.width, &mut dec)?;
                    Self::do_block4x4(&mut strip[x + 4 * self.width + 4..], self.width, &mut dec)?;
                }
            }
        }
        Ok(())
    }
}

impl InputSource for PMMDecoder {
    fn get_num_streams(&self) -> usize { if self.adec.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    15,
                    tb_num: 256,
                    tb_den: u32::from(self.fps),
                }),
            1 if self.adec.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.adec.arate),
                    sample_type: AudioSample::S16,
                    channels:    self.adec.channels,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = EndianReader { br: &mut self.fr, is_be: self.is_be };
        while br.tell() < self.end {
            if self.aframe && self.adec.audio.len() >= self.adec.ablk {
                let mut frame = vec![0; self.adec.ablk];
                frame.copy_from_slice(&self.adec.audio[..self.adec.ablk]);
                self.adec.audio.drain(..self.adec.ablk);
                self.aframe = false;
                return Ok((1, Frame::AudioS16(frame)));
            }
            let tag     = br.read_tag()?;
            let size    = br.read_u32()? as usize;
            validate!(br.tell() + (size as u64) <= self.end);
            match &tag {
                b"AUDO" if self.adec.arate > 0 => {
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.adec.unpack_audio(&self.data, self.is_be)?;
                },
                b"BODY" if size > 0 => {
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.unpack_video().map_err(|_| DecoderError::InvalidData)?;
                    self.aframe = true;
                    let mut frm = vec![0; self.width * self.height];
                    if !self.is_yuv {
                        for (dst, &pix) in frm.iter_mut().zip(self.frame.iter()) {
                            *dst = (pix >> 10) | (pix & 0x3E0) | ((pix & 0x1F) << 10);
                        }
                    } else {
                        for (dst, &src) in frm.iter_mut().zip(self.frame.iter()) {
                            *dst = self.yuv2rgb.remap[usize::from(src & 0x7FFF)];
                        }
                    }
                    return Ok((0, Frame::VideoRGB16(frm)));
                },
                b"CLUT" => {
                    validate!((size & 1) == 0 && (2..=512).contains(&size));
                    for el in self.pal.iter_mut().take(size / 2) {
                        *el = br.br.read_u16le()?;
                    }
                },
                b"FRAM" => {
                    validate!(size > 12);
                    let width = br.read_u32()? as usize;
                    let height = br.read_u32()? as usize;
                    let _smth = br.read_u32()?;
                    validate!(width > 0 && width <= self.width && height > 0 && height <= self.height);
                    validate!(size == width * height + 12);
                    for line in self.frame.chunks_exact_mut(self.width).take(height) {
                        for pix in line[..width].iter_mut() {
                            let idx = usize::from(br.read_byte()?);
                            *pix = self.pal[idx];
                        }
                    }
                    self.aframe = true;
                    return Ok((0, Frame::VideoRGB16(self.frame.clone())));
                },
                b"YCCF" => {
                    self.is_yuv = true;
                    match size {
                        0x2C => {
                            for el in self.yuv2rgb.coeffs.iter_mut() {
                                *el = f64::from(br.read_f32()?);
                            }
                            br.read_u32()?; // luma bias
                            br.read_u32()?; // chroma bias
                        },
                        0x48 => {
                            for el in self.yuv2rgb.coeffs.iter_mut() {
                                *el = br.read_f64()?;
                            }
                        },
                        _ => {
                            println!("Unknown YUV coefficients size {size}, colours may be wrong");
                        }
                    }
                    self.yuv2rgb.init();
                },
                _ => {
                    br.read_skip(size)?;
                },
            }
        }
        Err(DecoderError::EOF)
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"PIFF" || &tag == b"BIFF");
    let end_le = u64::from(fr.peek_u32le()?) + 8;
    let end_be = u64::from(fr.read_u32be()?) + 8;

    let tag = fr.read_tag()?;
    validate!(&tag == b"PXMT");
    let tag = fr.read_tag()?;
    validate!(&tag == b"PXMH");
    let size_le = fr.peek_u32le()? as usize;
    let size_be = fr.read_u32be()? as usize;
    validate!(size_be > 0);
    let is_be = size_be < 0x1000;
    let size = if is_be { size_be } else { size_le };
    let end = if is_be { end_be } else { end_le };

    validate!(size >= 0x38);

    let mut br = EndianReader { br: &mut fr, is_be };

    let version = br.read_u16()?;
    let _flags = br.read_u16()?;
    let width = usize::from(br.read_u16()?);
    let height = usize::from(br.read_u16()?);
    validate!((1..=80).contains(&width) && (1..=60).contains(&height));
    let bwidth = br.read_u16()?;
    let bheight = br.read_u16()?;
    validate!(bwidth == 8 && bheight == 8);
    let width = width * 8;
    let height = height * 8;
    let fps = br.read_u16()?;
    validate!((0x100..=0x1E00).contains(&fps));
    let nframes = br.read_u16()?;
    validate!(nframes > 0);
    br.read_u32()?; // maximum frame size
    br.read_skip(32)?; // description
    let arate = br.read_u16()?;
    let atype = br.read_byte()?;
    let channels = br.read_byte()?;
    validate!((arate == 0 && channels == 0) || (arate > 0 && (channels == 1 || channels == 2)));
    br.read_skip(size - 0x38)?;

    Ok(Box::new(PMMDecoder {
        fr, end,
        is_be,
        is_yuv: version >= 0x200,
        width, height, fps,
        data: Vec::new(),
        frame: vec![0; width * height],
        adec: AudioDecoder {
            arate, atype, channels,
            audio: Vec::new(),
            apred: [0; 2],
            ablk:  (usize::from(arate) * 256 / usize::from(fps)).max(usize::from(channels))
        },
        aframe: false,
        yuv2rgb: YUV2RGB::new(),
        pal: [0; 256],
    }))
}

const PCM16_TABLE: [i16; 256] = [
-32767,-30191,-27818,-25631,-23617,-21760,-20050,-18474,-17021,-15683,-14451,-13315,-12268,-11304,-10415, -9596,
 -8842, -8147, -7506, -6916, -6373, -5872, -5410, -4985, -4593, -4232, -3899, -3593, -3310, -3050, -2810, -2589,
 -2386, -2198, -2025, -1866, -1719, -1584, -1460, -1345, -1239, -1142, -1052,  -969,  -893,  -823,  -758,  -698,
  -643,  -593,  -546,  -503,  -464,  -427,  -393,  -363,  -334,  -308,  -283,  -261,  -241,  -222,  -204,  -188,
  -173,  -160,  -147,  -135,  -125,  -115,  -106,   -97,   -90,   -83,   -76,   -70,   -65,   -59,   -55,   -50,
   -46,   -43,   -39,   -36,   -33,   -31,   -28,   -26,   -24,   -22,   -20,   -19,   -17,   -16,   -14,   -13,
   -12,   -11,   -10,    -9,    -9,    -8,    -7,    -7,    -6,    -6,    -5,    -5,    -4,    -4,    -4,    -3,
    -3,    -3,    -2,    -2,    -2,    -2,    -2,    -1,    -1,    -1,    -1,    -1,    -1,    -1,    -1,     0,
     1,     1,     1,     1,     1,     1,     1,     1,     2,     2,     2,     2,     2,     3,     3,     3,
     4,     4,     4,     5,     5,     6,     6,     7,     7,     8,     9,     9,    10,    11,    12,    13,
    14,    16,    17,    19,    20,    22,    24,    26,    28,    31,    33,    36,    39,    43,    46,    50,
    55,    59,    65,    70,    76,    83,    90,    97,   106,   115,   125,   135,   147,   160,   173,   188,
   204,   222,   241,   261,   283,   308,   334,   363,   393,   427,   464,   503,   546,   593,   643,   698,
   758,   823,   893,   969,  1052,  1142,  1239,  1345,  1460,  1584,  1719,  1866,  2025,  2198,  2386,  2589,
  2810,  3050,  3310,  3593,  3899,  4232,  4593,  4985,  5410,  5872,  6373,  6916,  7506,  8147,  8842,  9596,
 10415, 11304, 12268, 13315, 14451, 15683, 17021, 18474, 20050, 21760, 23617, 25631, 27818, 30191, 32767, 32767
];

const PIX8_TAB0: [u16; 97] = [
 0x0000, 0x0400, 0x0800, 0x0C00, 0x1000, 0x1400, 0x1800, 0x1C00,
 0x2000, 0x2400, 0x2800, 0x2C00, 0x3000, 0x3400, 0x3800, 0x3C00,
 0x4000, 0x4400, 0x4800, 0x4C00, 0x5000, 0x5400, 0x5800, 0x5C00,
 0x6000, 0x6400, 0x6800, 0x6C00, 0x7000, 0x7400, 0x7800, 0x7C00,
 0x0000, 0x0020, 0x0040, 0x0060, 0x0080, 0x00A0, 0x00C0, 0x00E0,
 0x0100, 0x0120, 0x0140, 0x0160, 0x0180, 0x01A0, 0x01C0, 0x01E0,
 0x0200, 0x0220, 0x0240, 0x0260, 0x0280, 0x02A0, 0x02C0, 0x02E0,
 0x0300, 0x0320, 0x0340, 0x0360, 0x0380, 0x03A0, 0x03C0, 0x03E0,
 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
 0x0008, 0x0009, 0x000A, 0x000B, 0x000C, 0x000D, 0x000E, 0x000F,
 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
 0x0018, 0x0019, 0x001A, 0x001B, 0x001C, 0x001D, 0x001E, 0x001F,
 0x0000
];

const PIX8_TAB1: [u16; 97] = [
 0x0000, 0x0021, 0x0042, 0x0063, 0x0084, 0x00A5, 0x00C6, 0x00E7,
 0x0108, 0x0129, 0x014A, 0x016B, 0x018C, 0x01AD, 0x01CE, 0x01EF,
 0x0210, 0x0231, 0x0252, 0x0273, 0x0294, 0x02B5, 0x02D6, 0x02F7,
 0x0318, 0x0339, 0x035A, 0x037B, 0x039C, 0x03BD, 0x03DE, 0x03FF,
 0x0000, 0x0020, 0x0040, 0x0060, 0x0080, 0x00A0, 0x00C0, 0x00E0,
 0x0100, 0x0120, 0x0140, 0x0160, 0x0180, 0x01A0, 0x01C0, 0x01E0,
 0x0200, 0x0220, 0x0240, 0x0260, 0x0280, 0x02A0, 0x02C0, 0x02E0,
 0x0300, 0x0320, 0x0340, 0x0360, 0x0380, 0x03A0, 0x03C0, 0x03E0,
 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
 0x0008, 0x0009, 0x000A, 0x000B, 0x000C, 0x000D, 0x000E, 0x000F,
 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
 0x0018, 0x0019, 0x001A, 0x001B, 0x001C, 0x001D, 0x001E, 0x001F,
 0x0000
];

const PIX8_TAB2: [u16; 97] = [
 0x0000, 0x0400, 0x0800, 0x0C00, 0x1000, 0x1400, 0x1800, 0x1C00,
 0x2000, 0x2400, 0x2800, 0x2C00, 0x3000, 0x3400, 0x3800, 0x3C00,
 0x4000, 0x4400, 0x4800, 0x4C00, 0x5000, 0x5400, 0x5800, 0x5C00,
 0x6000, 0x6400, 0x6800, 0x6C00, 0x7000, 0x7400, 0x7800, 0x7C00,
 0x0000, 0x0401, 0x0802, 0x0C03, 0x1004, 0x1405, 0x1806, 0x1C07,
 0x2008, 0x2409, 0x280A, 0x2C0B, 0x300C, 0x340D, 0x380E, 0x3C0F,
 0x4010, 0x4411, 0x4812, 0x4C13, 0x5014, 0x5415, 0x5816, 0x5C17,
 0x6018, 0x6419, 0x681A, 0x6C1B, 0x701C, 0x741D, 0x781E, 0x7C1F,
 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
 0x0008, 0x0009, 0x000A, 0x000B, 0x000C, 0x000D, 0x000E, 0x000F,
 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
 0x0018, 0x0019, 0x001A, 0x001B, 0x001C, 0x001D, 0x001E, 0x001F,
 0x0000
];

const PIX8_TAB3: [u16; 97] = [
 0x0000, 0x0400, 0x0800, 0x0C00, 0x1000, 0x1400, 0x1800, 0x1C00,
 0x2000, 0x2400, 0x2800, 0x2C00, 0x3000, 0x3400, 0x3800, 0x3C00,
 0x4000, 0x4400, 0x4800, 0x4C00, 0x5000, 0x5400, 0x5800, 0x5C00,
 0x6000, 0x6400, 0x6800, 0x6C00, 0x7000, 0x7400, 0x7800, 0x7C00,
 0x0000, 0x0020, 0x0040, 0x0060, 0x0080, 0x00A0, 0x00C0, 0x00E0,
 0x0100, 0x0120, 0x0140, 0x0160, 0x0180, 0x01A0, 0x01C0, 0x01E0,
 0x0200, 0x0220, 0x0240, 0x0260, 0x0280, 0x02A0, 0x02C0, 0x02E0,
 0x0300, 0x0320, 0x0340, 0x0360, 0x0380, 0x03A0, 0x03C0, 0x03E0,
 0x0000, 0x0420, 0x0840, 0x0C60, 0x1080, 0x14A0, 0x18C0, 0x1CE0,
 0x2100, 0x2520, 0x2940, 0x2D60, 0x3180, 0x35A0, 0x39C0, 0x3DE0,
 0x4200, 0x4620, 0x4A40, 0x4E60, 0x5280, 0x56A0, 0x5AC0, 0x5EE0,
 0x6300, 0x6720, 0x6B40, 0x6F60, 0x7380, 0x77A0, 0x7BC0, 0x7FE0,
 0x0000
];

const PIX8_TABS: [&[u16; 97]; 8] = [
    &PIX8_TAB0, &PIX8_TAB0, &PIX8_TAB0, &PIX8_TAB1,
    &PIX8_TAB0, &PIX8_TAB2, &PIX8_TAB3, &PIX8_TAB0
];

const PIX4_TAB0: [u16; 97] = [
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FFF
];

const PIX4_TAB1: [u16; 97] = [
 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00,
 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00,
 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00,
 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00, 0x7C00,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FFF
];

const PIX4_TAB2: [u16; 97] = [
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0,
 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0,
 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0,
 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0, 0x03E0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0, 0x7FE0,
 0x7FFF
];

const PIX4_TAB3: [u16; 97] = [
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF, 0x03FF,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F, 0x7C1F,
 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F,
 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F,
 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F,
 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F, 0x001F,
 0x7FFF
];

const PIX4_TABS: [&[u16; 97]; 8] = [
    &PIX4_TAB0, &PIX4_TAB0, &PIX4_TAB0, &PIX4_TAB1,
    &PIX4_TAB0, &PIX4_TAB2, &PIX4_TAB3, &PIX4_TAB0
];
