use std::fs::File;
use std::collections::VecDeque;
use crate::io::byteio::*;
use super::super::*;

// ticks per frame for a fraction of 60 fps
const TICK_DIVISOR: u32 = 5;

struct NXLDecoder {
    fr:         FileReader<File>,
    flavour:    u8,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    tmp_pic:    Vec<u8>,
    line:       Vec<u8>,
    width:      usize,
    height:     usize,
    depth:      u16,
    arate:      u32,
    audio:      Vec<u8>,
    frames:     VecDeque<(u32, Frame)>,
    asize:      usize,
    apts:       u32,
    vpts:       u32,
    flush:      bool,
}

impl NXLDecoder {
    fn get_audio(&mut self) -> DecoderResult<(usize, Frame)> {
        let chunk_size = (self.arate * TICK_DIVISOR / 60) as usize;
        self.apts += chunk_size as u32;
        let ret = if self.audio.len() > chunk_size {
                let mut frm = vec![0; chunk_size];
                frm.copy_from_slice(&self.audio[..chunk_size]);
                self.audio.drain(..chunk_size);
                frm
            } else {
                let mut ret = Vec::new();
                std::mem::swap(&mut ret, &mut self.audio);
                ret
            };
        Ok((1, Frame::AudioU8(ret)))
    }
}

impl InputSource for NXLDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: TICK_DIVISOR,
                    tb_den: 60,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        loop {
            match (self.flush, !self.audio.is_empty(), !self.frames.is_empty()) {
                (_, true, true) => {
                    let audio_pts = u64::from(self.apts) * 60;
                    let video_pts = u64::from(self.frames[0].0) * u64::from(self.arate);
                    if audio_pts < video_pts {
                        return self.get_audio();
                    } else {
                        let (_, frm) = self.frames.pop_front().unwrap();
                        return Ok((0, frm));
                    }
                },
                (true, true, false) => return self.get_audio(),
                (true, false, true) => {
                    let (_, frm) = self.frames.pop_front().unwrap();
                    return Ok((0, frm));
                },
                (true, false, false) => return Err(DecoderError::EOF),
                _ => {},
            }

            let chunk_type = br.read_byte()?;
            br.read_skip(3)?;
            let size = br.read_u32be()? as usize;
            validate!(size >= 64);
            let _chunk_num = br.read_u32be()?;
            let time = br.read_u32be()?;

            match chunk_type {
                1 => return Err(DecoderError::InvalidData),
                3 => {
//                    validate!(time > self.vpts || self.vpts == 0);
                    let width = br.read_u16be()? as usize;
                    let height = br.read_u16be()? as usize;
                    let x_off = br.read_u16be()? as usize;
                    let y_off = br.read_u16be()? as usize;
                    let scaling = br.read_byte()?;
                    validate!(scaling < 5);
                    let (scaled_w, scaled_h) = match scaling {
                            0 => (width, height),
                            1 => (width * 2, height),
                            2 => (width, height * 2),
                            3 => (width * 2, height * 2),
                            4 => (width, height * 2),
                            _ => unreachable!(),
                        };
                    validate!(x_off + scaled_w <= self.width && y_off + scaled_h <= self.height);
                    let _flags = br.read_byte()?;
                    let _nclrs = br.read_u16be()? as usize;
                    let mut vsize = br.read_u32be()? as usize;
                    let pal_size = 3 << self.depth;
                    // sometimes video part size is set to a larger size than it really is
                    if self.depth == 8 {
                        let raw_size = width * height;
                        if vsize > raw_size {
                            vsize = raw_size;
                        }
                    }
                    validate!(vsize + pal_size <= size - 64);
                    let seg_size = (width + 7) / 8;
                    validate!(vsize == height * seg_size * (self.depth as usize));
                    br.read_skip(32)?;

                    // this seems to be some kind of alternative video stream embedded without any warning
                    if width == 64 && height == 64 && x_off == 0 && y_off == 0 {
                        br.read_skip(size - 64)?;
                        continue;
                    }

                    validate!(time >= self.vpts);

                    let is_mac_nxl2 = self.depth == 8 && self.flavour == 0x14;

                    if !is_mac_nxl2 {
                        br.read_vga_pal_some(&mut self.pal[..pal_size])?;
                    } else {
                        for clr in self.pal[..pal_size].chunks_exact_mut(3) {
                            let b = br.read_u16le()?;
                            clr[2] = (b >> 10 << 3) as u8;
                            clr[1] = (b >>  5 << 3) as u8;
                            clr[0] = (b       << 3) as u8;
                        }
                    }

                    if width > 0 {
                        self.tmp_pic.clear();
                        self.tmp_pic.resize(width * height, 0);
                        self.line.resize(seg_size, 0);
                        if self.depth < 8 {
                            for dline in self.tmp_pic.chunks_exact_mut(width) {
                                for plane in 0..(self.depth as u8) {
                                    br.read_buf(&mut self.line)?;
                                    add_line(dline, &self.line, plane);
                                }
                            }
                            blit_frame_ega(&mut self.frame[x_off + y_off * self.width..], self.width,
                                           &self.tmp_pic, width, scaling);
                        } else {
                            for dline in self.tmp_pic.chunks_exact_mut(width) {
                                br.read_buf(dline)?;
                            }
                            if !is_mac_nxl2 {
                                blit_frame_vga(&mut self.frame[x_off + y_off * self.width..], self.width,
                                               &self.tmp_pic, width, scaling);
                            } else {
                                // skip leftover after palette
                                br.read_skip(256)?;
                                blit_frame_ega(&mut self.frame[x_off + y_off * self.width..], self.width,
                                               &self.tmp_pic, width, scaling);

                            }
                        }
                    }
                    br.read_skip(size - 64 - pal_size - vsize)?;

                    let start_ts = (self.vpts + TICK_DIVISOR - 1) / TICK_DIVISOR * TICK_DIVISOR;
                    if start_ts < time {
                        for pts in (start_ts..time).step_by(TICK_DIVISOR as usize) {
                            let pal = self.pal;
                            let frame = self.frame.clone();
                            self.frames.push_back((pts, Frame::VideoPal(frame, pal)));
                        }
                    }
                    self.vpts = time;
                },
                4 => {
                    let arate = u32::from(br.read_u16be()?);
                    let asize = br.read_u16be()? as usize;
                    validate!(arate == self.arate);
                    validate!(asize <= size - 64);
                    br.read_skip(44)?;

                    self.asize += asize;
                    br.read_extend(&mut self.audio, asize)?;
                    br.read_skip(size - 64 - asize)?;
                },
                0xFF => self.flush = true,
                _ => br.read_skip(size - 16)?,
            }
        }
    }
}

fn add_line(dline: &mut [u8], sline: &[u8], shift: u8) {
    for (dst, &src) in dline.chunks_mut(8).zip(sline.iter()) {
        let mut byte = src;
        for el in dst.iter_mut() {
            *el |= byte >> 7 << shift;
            byte <<= 1;
        }
    }
}

fn blit_frame_ega(dst: &mut [u8], dstride: usize, src: &[u8], sstride: usize, scaling: u8) {
    match scaling {
        0 => {
            for (dline, sline) in dst.chunks_mut(dstride).zip(src.chunks_exact(sstride)) {
                dline[..sstride].copy_from_slice(sline);
            }
        },
        1 => {
            for (dline, sline) in dst.chunks_mut(dstride).zip(src.chunks_exact(sstride)) {
                for (pair, &src) in dline.chunks_exact_mut(2).zip(sline.iter()) {
                    pair[0] = src;
                    pair[1] = src;
                }
            }
        },
        2 => {
            for (dlines, sline) in dst.chunks_mut(dstride * 2).zip(src.chunks_exact(sstride)) {
                let (dline0, dline1) = dlines.split_at_mut(dstride);
                dline0[..sstride].copy_from_slice(sline);
                dline1[..sstride].copy_from_slice(sline);
            }
        },
        3 => {
            for (dlines, sline) in dst.chunks_mut(dstride * 2).zip(src.chunks_exact(sstride)) {
                let (dline0, dline1) = dlines.split_at_mut(dstride);
                for ((dst0, dst1), &src) in dline0.chunks_exact_mut(2)
                        .zip(dline1.chunks_exact_mut(2)).zip(sline.iter()) {
                    dst0[0] = src;
                    dst0[1] = src;
                    dst1[0] = src;
                    dst1[1] = src;
                }
            }
        },
        4 => {
            for (dlines, sline) in dst.chunks_mut(dstride * 2).zip(src.chunks_exact(sstride)) {
                dlines[..sstride].copy_from_slice(sline);
            }
        },
        _ => unimplemented!(),
    }
}

fn blit_frame_vga(dst: &mut [u8], dstride: usize, src: &[u8], sstride: usize, scaling: u8) {
    match scaling {
        0 => { // four stripes, no scaling
            for (dline, sline) in dst.chunks_mut(dstride).zip(src.chunks_exact(sstride)) {
                let (first, second) = sline.split_at(sline.len() / 2);
                let (q0, q1) = first.split_at(sline.len() / 4);
                let (q2, q3) = second.split_at(sline.len() / 4);
                for (quad, ((&s0, &s1), (&s2, &s3))) in dline.chunks_exact_mut(4)
                        .zip(q0.iter().zip(q1.iter()).zip(q2.iter().zip(q3.iter()))) {
                    quad[0] = s0;
                    quad[1] = s1;
                    quad[2] = s2;
                    quad[3] = s3;
                }
            }
        },
        1 => { // two stripes -> interleave scale 2x horizontally
            for (dline, sline) in dst.chunks_mut(dstride).zip(src.chunks_exact(sstride)) {
                let (first, second) = sline.split_at(sstride / 2);
                for (quad, (&src0, &src1)) in dline.chunks_exact_mut(4)
                        .zip(first.iter().zip(second.iter())) {
                    quad[0] = src0;
                    quad[1] = src0;
                    quad[2] = src1;
                    quad[3] = src1;
                }
            }
        },
        2 => { // four stripes -> interleave and scale 2x vertically
            for (dlines, sline) in dst.chunks_mut(dstride * 2).zip(src.chunks_exact(sstride)) {
                let (dline0, dline1) = dlines.split_at_mut(dstride);
                let (first, second) = sline.split_at(sline.len() / 2);
                let (q0, q1) = first.split_at(sline.len() / 4);
                let (q2, q3) = second.split_at(sline.len() / 4);
                for ((quad0, quad1), ((&s0, &s1), (&s2, &s3))) in dline0.chunks_exact_mut(4)
                        .zip(dline1.chunks_exact_mut(4))
                        .zip(q0.iter().zip(q1.iter()).zip(q2.iter().zip(q3.iter()))) {
                    quad0[0] = s0;
                    quad0[1] = s1;
                    quad0[2] = s2;
                    quad0[3] = s3;
                    quad1[0] = s0;
                    quad1[1] = s1;
                    quad1[2] = s2;
                    quad1[3] = s3;
                }
            }
        },
        _ => unimplemented!(),
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    let b = br.read_byte()?;
    validate!(b == 1);
    let flavour = br.read_byte()?;
    br.read_skip(2)?;
    let size = br.read_u32be()?;
    validate!(size == 64);
    let chunk_no = br.read_u32be()?;
    validate!(chunk_no == 0);
    br.read_skip(4)?;
    let tag = br.read_tag()?;
    validate!(&tag == b"NXL1" || &tag == b"NXL2");
    let width = br.read_u16be()? as usize;
    let height = br.read_u16be()? as usize;
    validate!(width > 0 && width <= 640 && height > 0 && height <= 480);
    let depth = br.read_u16be()?;
    validate!(depth > 0 && depth <= 8);
    br.read_skip(4)?;
    br.read_u32be()?;
    let arate = br.read_u32be()?;
    let _aflags = br.read_u16be()?;
    br.read_skip(24)?;

    Ok(Box::new(NXLDecoder {
        fr: br,
        flavour,
        width, height, depth,
        arate,
        pal:     [0; 768],
        frame:   vec![0; width * height],
        tmp_pic: Vec::with_capacity(width * height),
        line:    Vec::with_capacity(width),
        apts:    0,
        vpts:    0,
        audio:   Vec::new(),
        frames:  VecDeque::new(),
        asize:   0,
        flush:   false,
    }))
}
