use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct NAFDecoder {
    fr:         FileReader<File>,
    width:      usize,
    height:     usize,
    pal:        [u8; 768],
}

impl InputSource for NAFDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        self.fr.peek_byte().map_err(|_| DecoderError::EOF)?;
        let mut frm = vec![0; self.width * self.height];
        self.fr.read_buf(&mut frm)?;
        Ok((0, Frame::VideoPal(frm, self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let n = fr.read_byte()?;
    let x = fr.read_byte()?;
    let f = fr.read_byte()?;
    validate!(n == b'N' && (x == b'A' || x == b'G') && f == b'F');
    let is_video = x == b'A';
    let one = fr.read_byte()?;
    let zero = fr.read_byte()?;
    validate!(one == 1 && zero == 0);
    let width = usize::from(fr.read_u16le()?);
    let height = usize::from(fr.read_u16le()?);
    let mut pal = [0; 768];
    if is_video {
        validate!((1..=320).contains(&width) && (1..=240).contains(&height));
        fr.read_vga_pal(&mut pal)?;
    } else {
        validate!((1..=640).contains(&width) && (1..=480).contains(&height));
        fr.read_buf(&mut pal)?;
    }

    Ok(Box::new(NAFDecoder { fr, width, height, pal }))
}
