use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 640;
const HEIGHT: usize = 480;

struct MPAnimDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    fnum:       u32,
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    is_seq:     bool,
}

impl MPAnimDecoder {
    fn unpack_delta(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let mut dpos = 0;
        loop {
            let op = br.read_byte()?;
            match op {
                0xD3 if !self.is_seq => {
                    let run = usize::from(br.read_byte()?);
                    let clr = br.read_byte()?;
                    validate!(dpos + run <= self.frame.len());
                    for dst in self.frame[dpos..][..run].iter_mut() {
                        *dst = clr;
                    }
                    dpos += run;
                },
                0xD4..=0xDD if !self.is_seq => {
                    let run = usize::from(op - 0xD3);
                    let clr = br.read_byte()?;
                    validate!(dpos + run <= self.frame.len());
                    for dst in self.frame[dpos..][..run].iter_mut() {
                        *dst = clr;
                    }
                    dpos += run;
                },
                0xDE..=0xFA if !self.is_seq => {
                    let skip = usize::from(op - 0xDD);
                    dpos += skip;
                },
                0xFB => return Err(DecoderError::NotImplemented),
                0xFC => return Ok(()),
                0xFD => {
                    let skip = usize::from(br.read_byte()?);
                    dpos += skip;
                },
                0xFE => {
                    let skip = usize::from(br.read_u16le()?);
                    dpos += skip;
                },
                0xFF => {
                    let skip = br.read_u32le()? as usize;
                    dpos += skip;
                },
                _ => {
                    validate!(dpos < self.frame.len());
                    self.frame[dpos] = op;
                    dpos += 1;
                },
            }
        }
    }
}

impl InputSource for MPAnimDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: if !self.is_seq { 12 } else { 25 },
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.fnum > 0 {
            let mut tag = [0; 6];
            self.fr.read_buf(&mut tag).map_err(|_| DecoderError::EOF)?;
            validate!(&tag == b"IMAGE=");
            let _frameno = self.fr.read_u16le()?;
            let size = self.fr.read_u32le()? as usize;
            validate!((1..=WIDTH * HEIGHT * 2).contains(&size));
            self.fr.read_u32le()?;
            self.vdata.resize(size, 0);
            self.fr.read_buf(&mut self.vdata)?;
            self.unpack_delta().map_err(|_| DecoderError::InvalidData)?;
        }
        self.fnum += 1;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str, is_seq: bool) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut tag = [0; 6];
    fr.read_buf(&mut tag)?;
    validate!(&tag == b"MPANIM");
    let mut pal = [0; 768];
    fr.read_buf(&mut pal)?;
    fr.read_skip(32)?;

    let tag = fr.read_tag()?;
    validate!(&tag == b"NORM");
    let norm_size = fr.read_u32le()? as usize;
    validate!(norm_size == WIDTH * HEIGHT);
    fr.read_skip(14)?;
    let mut frame = vec![0; WIDTH * HEIGHT];
    fr.read_buf(&mut frame)?;

    Ok(Box::new(MPAnimDecoder {
        fr,
        is_seq,
        pal, frame,
        fnum: 0,
        vdata: Vec::with_capacity(WIDTH * HEIGHT),
    }))
}

pub fn open_anm(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open(name, false)
}

pub fn open_seq(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open(name, true)
}
