use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const MAX_BITS:   u8 = 13;
const DICT_SIZE: usize = 1 << MAX_BITS;
const INVALID_POS: usize = 65536;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
    nsyms:      usize,
    idx_bits:   u8,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
            idx_bits:   0,
            nsyms:      0,
        }
    }
    fn reset(&mut self, bits: u8) {
        self.nsyms    = (1 << bits) + 2;
        self.dict_pos = self.nsyms;
        self.dict_lim = 1 << (bits + 1);
        self.idx_bits = bits + 1;
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut [u8], pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        validate!(pos + tot_len <= dst.len());

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], dst: &mut [u8]) -> DecoderResult<usize> {
        let mut br = BitReader::new(src, BitReaderMode::LE);

        let bits = 8;
        let reset_sym = 1 << bits;
        let end_sym = reset_sym + 1;

        self.reset(bits);

        let mut pos = 0;
        let mut lastidx = INVALID_POS;
        let mut prev = 0u8;
        loop {
            let idx         = br.read(self.idx_bits)? as usize;
            if idx == reset_sym {
                self.reset(bits);
                lastidx = INVALID_POS;
                prev = 0;
                continue;
            }
            if idx == end_sym {
                break;
            }
            validate!(idx <= self.dict_pos);
            let lastpos = pos;
            if idx != self.dict_pos {
                let len = self.decode_idx(dst, pos, idx)?;
                if lastidx != INVALID_POS {
                    self.add(lastidx, dst[pos]);
                }
                pos += len;
            } else {
                validate!(lastidx != INVALID_POS);
                let len = self.decode_idx(dst, pos, lastidx)?;
                let lastsym = dst[pos];
                pos += len;
                validate!(pos < dst.len());
                dst[pos] = lastsym;
                pos += 1;
                self.add(lastidx, lastsym);
            }
            for el in dst[lastpos..pos].iter_mut() {
                prev = prev.wrapping_add(*el);
                *el = prev;
            }

            lastidx = idx;
            if self.dict_pos + 1 == self.dict_lim && self.idx_bits < MAX_BITS {
                self.dict_lim <<= 1;
                self.idx_bits += 1;
            }
        }
        validate!(pos <= dst.len());
        Ok(pos)
    }
}

const WIDTH:  usize = 320;
const HEIGHT: usize = 180;

struct MgifDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    cchunk:     u32,
    nchunks:    u32,
    frame:      [u16; WIDTH * HEIGHT],
    udata:      [u8; 64000],
    pal:        [u16; 256],
    lzw:        LZWState,
    abuf:       Vec<u8>,
}

fn recon_inter(frm: &mut [u16; WIDTH * HEIGHT], src: &[u8], pal: &[u16; 256]) -> DecoderResult<()> {
    validate!(src.len() > 4);
    let (hdr, data) = src.split_at(4);
    let part2_off = read_u32le(hdr).unwrap_or_default() as usize;
    validate!(part2_off <= data.len());
    let (ops_data, pixels_data) = data.split_at(part2_off);
    let mut ops = MemoryReader::new_read(ops_data);
    let mut pixels = MemoryReader::new_read(pixels_data);

    let mut x = 0;
    let mut y = 0;
    while y < HEIGHT {
        let op = ops.read_byte()?;
        if (op & 0xC0) != 0xC0 {
            x += usize::from(op) * 2;
            let mut copy_len = usize::from(ops.read_byte()?) * 2;
            if copy_len == 0 {
                copy_len = 256 * 2;
            }
            validate!(x + copy_len <= WIDTH);
            for pix in frm[x + y * WIDTH..][..copy_len].iter_mut() {
                *pix = pal[usize::from(pixels.read_byte()?)];
            }
            x += copy_len;
        } else {
            y += usize::from(op & 0x3F) + 1;
            x = 0;
        }
    }

    Ok(())
}

impl InputSource for MgifDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    15,
                tb_num: 1,
                tb_den: 15,
            }),
            1 => StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::U8,
                channels:    2,
            }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut got_frame = false;
        let mut frame_type = 0;
        let mut frame_len = 0;
        while self.cchunk < self.nchunks {
            if !self.abuf.is_empty() {
                let mut ret = Vec::new();
                std::mem::swap(&mut ret, &mut self.abuf);
                return Ok((1, Frame::AudioU8(ret)));
            }
            self.cchunk += 1;

            let nchunks = usize::from(self.fr.read_byte()?);
            let clen = self.fr.read_u24le()?;
            let chunk_end = self.fr.tell() + u64::from(clen);
            for _ in 0..nchunks {
                let ctype = self.fr.read_byte()?;
                let clen = self.fr.read_u24le()? as usize;
                match ctype {
                    1 | 2 => {
                        self.data.resize(clen, 0);
                        self.fr.read_buf(&mut self.data)?;
                        let len = self.lzw.unpack(&self.data, &mut self.udata)
                            .map_err(|_| DecoderError::InvalidData)?;
                        got_frame = true;
                        frame_type = ctype;
                        frame_len = len;
                    },
                    3 => {
                        validate!((2..=0x200).contains(&clen) && (clen & 1) == 0);
                        for clr in self.pal[..clen / 2].iter_mut() {
                            *clr = self.fr.read_u16le()?;
                        }
                    },
                    4 => {
                        if clen > 0 {
                            self.fr.read_extend(&mut self.abuf, clen)?;
                        }
                    },
                    _ => {
                        self.fr.read_skip(clen)?;
                    },
                }
            }
            validate!(self.fr.tell() == chunk_end);
            if got_frame {
                if frame_type == 1 {
                    validate!(frame_len == WIDTH * HEIGHT);
                    for (dst, &src) in self.frame.iter_mut().zip(self.udata.iter()) {
                        *dst = self.pal[usize::from(src)];
                    }
                } else {
                    recon_inter(&mut self.frame, &self.udata[..frame_len], &self.pal)
                        .map_err(|_| DecoderError::InvalidData)?;
                }
                return Ok((0, Frame::VideoRGB16(self.frame.to_vec())));
            }
        }
        Err(DecoderError::EOF)
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut tag = [0; 6];
    fr.read_buf(&mut tag)?;
    validate!(&tag == b"MGIF97");
    fr.read_skip(3)?;
    let nchunks = fr.read_u32le()?;
    validate!((1..=10000).contains(&nchunks));
    fr.seek(SeekFrom::Start(0x253))?;

    Ok(Box::new(MgifDecoder {
        fr,
        frame: [0; WIDTH * HEIGHT],
        udata: [0; 64000],
        pal: [0; 256],
        data: Vec::new(),
        cchunk: 0,
        nchunks,
        lzw: LZWState::new(),
        abuf: Vec::new(),
    }))
}
