use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const TILE_W: usize = 32;
const TILE_H: usize = 28;
const ABLK_SIZE: usize = 0x8B2;

struct KingpinBinDecoder {
    fr:         FileReader<BufReader<File>>,
    abuf:       Vec<u8>,
    audio:      bool,
    tiles:      Vec<[u8; 64]>,
    cram:       [u16; 64],
    c000:       [u8; 8192],
    e000:       [u8; 8192],
}

impl KingpinBinDecoder {
    fn render_frame(&mut self) -> Vec<u16> {
        const WIDTH: usize = TILE_W * 8;
        let mut frame = vec![0; WIDTH * TILE_H * 8];

        for (strip, row) in frame.chunks_exact_mut(WIDTH * 8).zip(self.e000.chunks_exact(32 * 2)) {
            for (x, src) in row.chunks_exact(2).enumerate().take(TILE_W) {
                let word = read_u16be(src).unwrap_or(0);
                let pal_idx = ((word >> 13) & 3) as usize;
                let _rev_v = (word & 0x1000) != 0;
                let _rev_h = (word & 0x0800) != 0;
                let tile_idx = (word & 0x7FF) as usize;

                let tile = &self.tiles[tile_idx];
                for (line, tline) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(8)) {
                    for (dst, &clr_idx) in line[x * 8..].iter_mut().zip(tline.iter()) {
                        if clr_idx != 0 {
                            *dst = self.cram[pal_idx * 16 + usize::from(clr_idx)];
                        }
                    }
                }
            }
        }

        for (strip, row) in frame.chunks_exact_mut(WIDTH * 8).zip(self.c000.chunks_exact(32 * 2)) {
            for (x, src) in row.chunks_exact(2).enumerate().take(TILE_W) {
                let word = read_u16be(src).unwrap_or(0);
                let pal_idx = ((word >> 13) & 3) as usize;
                let _rev_v = (word & 0x1000) != 0;
                let _rev_h = (word & 0x0800) != 0;
                let tile_idx = (word & 0x7FF) as usize;

                let tile = &self.tiles[tile_idx];
                for (line, tline) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(8)) {
                    for (dst, &clr_idx) in line[x * 8..].iter_mut().zip(tline.iter()) {
                        if clr_idx != 0 {
                            *dst = self.cram[pal_idx * 16 + usize::from(clr_idx)];
                        }
                    }
                }
            }
        }

        frame
    }
}

impl InputSource for KingpinBinDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  TILE_W * 8,
                    height: TILE_H * 8,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 10,
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22216,
                    sample_type: AudioSample::U8,
                    channels:    1,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        loop {
            if self.audio && self.abuf.len() >= ABLK_SIZE {
                let mut audio = vec![0; ABLK_SIZE];
                audio.copy_from_slice(&self.abuf[..ABLK_SIZE]);
                for el in audio.iter_mut() {
                    if *el < 0x80 {
                        *el = 0x80 - *el;
                    }
                }
                self.abuf.drain(..ABLK_SIZE);
                self.audio = false;
                return Ok((1, Frame::AudioU8(audio)));
            }
            let ctype = br.read_tag()?;
            let csize = br.read_u32be()? as usize;
            validate!(csize >= 8);
            match &ctype {
                b"AUDI" => {
                    br.read_extend(&mut self.abuf, csize - 8)?;
                },
                b"IVRA" => {
                    validate!((csize & 0xF) == 0xA && (csize & 0x10) == 0);
                    let offset = br.read_u16be()? as usize;
                    let ntiles = (csize - 10) / 32;
                    validate!((offset & 0x1F) == 0);
                    validate!(offset / 32 + ntiles <= self.tiles.len());
                    let mut raw_tile = [0; 32];
                    for tile in self.tiles[offset / 32..][..ntiles].iter_mut() {
                        br.read_buf(&mut raw_tile)?;
                        for (pair, &b) in tile.chunks_exact_mut(2).zip(raw_tile.iter()) {
                            pair[0] = b >> 4;
                            pair[1] = b & 0xF;
                        }
                    }
                },
                b"CRAM" => {
                    validate!(csize == 0x88);
                    for el in self.cram.iter_mut() {
                        let clr = br.read_u16be()?;
                        let b = (clr >> 8) & 0xF;
                        let g = (clr >> 4) & 0xF;
                        let r =  clr       & 0xF;
                        *el = (r << 11) | ((r >> 3) << 10)
                            | (g <<  6) | ((g >> 3) <<  5)
                            | (b <<  1) |  (b >> 3);
                    }
                },
                b"VRAM" => {
                    validate!(csize > 10 && (csize & 1) == 0);
                    let offset = br.read_u16be()? as usize;
                    validate!((0xC000..=0xEFFF).contains(&offset));
                    validate!((offset & 1) == 0);
                    validate!((offset & 0x1FFF) + csize - 10 <= 0x2000);
                    let dst = if offset < 0xE000 { &mut self.c000 } else { &mut self.e000 };
                    br.read_buf(&mut dst[offset & 0x1FFF..][..csize - 10])?;
                },
                b"CEND" => {
                    self.audio = true;
                    let frame = self.render_frame();
                    return Ok((0, Frame::VideoRGB16(frame)));
                }
                b"MEND" => return Err(DecoderError::EOF),
                _ => {
                    br.read_skip(csize - 8)?;
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    Ok(Box::new(KingpinBinDecoder {
        fr,
        abuf: Vec::new(),
        audio: false,
        tiles: vec![[0; 64]; 2048],
        cram: [0; 64],
        c000: [0; 8192],
        e000: [0; 8192],
    }))
}
