use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct KDVDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    cur_frm:    usize,
    offsets:    Vec<u32>,
}

impl KDVDecoder {
    // typical PCX RLE
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        for line in self.frame.chunks_exact_mut(self.width) {
            let mut pos = 0;
            while pos < line.len() {
                let op = br.read_byte()?;
                if (op & 0xC0) != 0xC0 {
                    line[pos] = op;
                    pos += 1;
                } else {
                    let len = usize::from(op & 0x3F);
                    let clr = br.read_byte()?;
                    validate!(pos + len <= line.len());
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                }
            }
        }

        Ok(())
    }
}

impl InputSource for KDVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 12,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm + 1 >= self.offsets.len() {
            return Err(DecoderError::EOF);
        }
        let br = &mut self.fr;
        let offset = self.offsets[self.cur_frm];
        let next_off = self.offsets[self.cur_frm + 1];
        self.cur_frm += 1;

        br.seek(SeekFrom::Start(offset.into()))?;
        self.vdata.resize((next_off - offset) as usize, 0);
        br.read_buf(&mut self.vdata)?;
        self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let magic = br.read_tag()?;
    validate!(&magic == b"KRL0");
    let full_size = br.read_u32le()? + 8;
    let magic = br.read_tag()?;
    validate!(&magic == b"KDV0");
    let size = br.read_u32le()? + 8;
    validate!(size == full_size);
    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let start = br.read_u32le()? + 8;
    let pal_start = br.read_u32le()? + 8;
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 320 && height > 0 && height <= 240);
    validate!(br.tell() + (nframes as u64) * 4 <= u64::from(pal_start));

    let mut offsets = Vec::with_capacity(nframes + 1);
    let off = br.read_u32le()? + 8;
    validate!(off == start);
    offsets.push(off);
    let mut prev_off = off;
    for _ in 1..nframes {
        let offset = br.read_u32le()? + 8;
        validate!(offset > prev_off);
        offsets.push(offset);
        prev_off = offset;
    }
    validate!(prev_off < full_size);
    offsets.push(full_size);
    br.seek(SeekFrom::Start(pal_start.into()))?;

    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    Ok(Box::new(KDVDecoder {
        fr: br,
        frame:      vec![0; width * height],
        vdata:      Vec::new(),
        width, height, pal,
        offsets,
        cur_frm: 0,
    }))
}
