use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct IVDDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u8,
    pal:        [u8; 768],
    arate:      u32,
    abuf:       Vec<u8>,
    ablk_len:   usize,
    audio:      bool,
    nframes:    u32,
    cur_frm:    u32,
}

impl IVDDecoder {
    fn draw_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);
        let mut last_clr = 0;
        let mut masks = [0; 32];
        let mut clrs = [0; 16];

        for strip in self.frame.chunks_exact_mut(self.width * 8) {
            for x in (0..self.width).step_by(8) {
                let op = br.read_byte()?;
                match op {
                    0x00 => {
                        for line in strip.chunks_exact_mut(self.width) {
                            br.read_buf(&mut line[x..][..8])?;
                        }
                    },
                    0x01..=0x10 => {
                        let pat = &PATTERNS[usize::from(op - 1)];
                        br.read_buf(&mut masks[..8])?;
                        for (i, coord) in pat.chunks_exact(2).enumerate() {
                            if ((masks[i >> 3] >> (7 - (i & 7))) & 1) != 0 {
                                last_clr = br.read_byte()?;
                            }
                            strip[x + usize::from(coord[1]) + usize::from(coord[0]) * self.width] = last_clr;
                        }
                    },
                    0x80 => {
                        br.read_buf(&mut masks[..8])?;
                        let clr = br.read_byte()?;
                        for (line, &mask) in strip.chunks_exact_mut(self.width).zip(masks.iter()) {
                            for (i, pix) in line[x..][..8].iter_mut().enumerate() {
                                *pix = if (mask >> (7 - i)) & 1 == 0 { clr } else { br.read_byte()? };
                            }
                        }
                    },
                    0x81 => {
                        br.read_buf(&mut masks[..8])?;
                        br.read_buf(&mut clrs[..2])?;
                        for (line, &mask) in strip.chunks_exact_mut(self.width).zip(masks.iter()) {
                            for (i, pix) in line[x..][..8].iter_mut().enumerate() {
                                *pix = if (mask >> (7 - i)) & 1 == 0 { clrs[0] } else { clrs[1] };
                            }
                        }
                    },
                    0xFB => {
                        br.read_buf(&mut clrs[..4])?;
                        br.read_buf(&mut masks[..16])?;
                        for (line, msk) in strip.chunks_exact_mut(self.width)
                                .zip(masks.chunks_exact(2)) {
                            let mut mask = usize::from(read_u16le(msk).unwrap_or_default());
                            for pix in line[x..][..8].iter_mut() {
                                *pix = clrs[mask & 3];
                                mask >>= 2;
                            }
                        }
                    },
                    0xFD => {
                        br.read_buf(&mut clrs[..16])?;
                        br.read_buf(&mut masks[..32])?;
                        for (line, msk) in strip.chunks_exact_mut(self.width)
                                .zip(masks.chunks_exact(4)) {
                            let mask = read_u32be(msk).unwrap_or_default() as usize;
                            for (i, pix) in line[x..][..8].iter_mut().enumerate() {
                                *pix = clrs[(mask >> ((28 - 4 * i) ^ 4)) & 0xF];
                            }
                        }
                    },
                    0xFF => {
                        let clr = br.read_byte()?;
                        for line in strip.chunks_exact_mut(self.width) {
                            for pix in line[x..][..8].iter_mut() {
                                *pix = clr;
                            }
                        }
                    }
                    _ => return Err(DecoderError::InvalidData),
                }
            }
        }

        Ok(())
    }
}

impl InputSource for IVDDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio && self.abuf.len() >= self.ablk_len {
            let mut ret = vec![0; self.ablk_len];
            ret.copy_from_slice(&self.abuf[..self.ablk_len]);
            self.abuf.drain(..self.ablk_len);
            self.audio = false;
            return Ok((1, Frame::AudioU8(ret)));
        }

        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }

        if self.arate > 0 {
            let asize = usize::from(self.fr.read_u16le()?);
            if asize > 0 && self.arate > 0 {
                self.fr.read_extend(&mut self.abuf, asize)?;
            } else {
                self.fr.read_skip(asize)?;
            }
        }
        let pal_size = usize::from(self.fr.read_u16le()?);
        validate!(pal_size <= self.pal.len());
        if pal_size > 0 {
            self.fr.read_vga_pal_some(&mut self.pal[768 - pal_size..])?;
        }
        let vsize = usize::from(self.fr.read_u16le()?);
        if vsize > 0 {
            self.vdata.resize(vsize, 0);
            self.fr.read_buf(&mut self.vdata)?;
            self.draw_frame().map_err(|_| DecoderError::InvalidData)?;
        }
        self.audio = true;
        self.cur_frm += 1;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let magic = fr.read_tag()?;
    validate!(&magic == b"VID\x01");
    let nframes = fr.read_u32le()?;
    validate!(nframes > 0);
    fr.read_u32le()?;
    let eight = fr.read_byte()?;
    validate!(eight == 8);
    let width = usize::from(fr.read_u16le()?);
    let height = usize::from(fr.read_u16le()?);
    validate!((8..=320).contains(&width) && (8..=240).contains(&height) && ((width | height) & 7) == 0);
    fr.read_byte()?;
    let fps = fr.read_byte()?.max(1);
    validate!((1..=30).contains(&fps));
    let nclrs = usize::from(fr.read_u16le()?);
    validate!((1..=256).contains(&nclrs));
    let arate = fr.read_u32le()?;
    fr.seek(SeekFrom::Start(0x100))?;
    let mut pal = [0; 768];
    fr.read_vga_pal_some(&mut pal[..nclrs * 3])?;

    Ok(Box::new(IVDDecoder {
        fr,
        width, height, fps, pal,
        nframes,
        cur_frm: 0,
        frame: vec![0; width * height],
        vdata: Vec::new(),
        arate,
        abuf: Vec::new(),
        audio: false,
        ablk_len: (arate as usize / usize::from(fps)).max(1),
    }))
}

static PATTERNS: [[u8; 8 * 8 * 2]; 16] = [
  [
    0, 0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 1, 0, 1, 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7,
    2, 0, 2, 1, 2, 2, 2, 3, 2, 4, 2, 5, 2, 6, 2, 7, 3, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3, 5, 3, 6, 3, 7,
    4, 0, 4, 1, 4, 2, 4, 3, 4, 4, 4, 5, 4, 6, 4, 7, 5, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5, 5, 5, 6, 5, 7,
    6, 0, 6, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 7, 7, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7,
  ], [
    0, 7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 0, 1, 7, 1, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1, 1, 1, 0,
    2, 7, 2, 6, 2, 5, 2, 4, 2, 3, 2, 2, 2, 1, 2, 0, 3, 7, 3, 6, 3, 5, 3, 4, 3, 3, 3, 2, 3, 1, 3, 0,
    4, 7, 4, 6, 4, 5, 4, 4, 4, 3, 4, 2, 4, 1, 4, 0, 5, 7, 5, 6, 5, 5, 5, 4, 5, 3, 5, 2, 5, 1, 5, 0,
    6, 7, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 6, 0, 7, 7, 7, 6, 7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0,
  ], [
    0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 0, 1, 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1,
    0, 2, 1, 2, 2, 2, 3, 2, 4, 2, 5, 2, 6, 2, 7, 2, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3, 5, 3, 6, 3, 7, 3,
    0, 4, 1, 4, 2, 4, 3, 4, 4, 4, 5, 4, 6, 4, 7, 4, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5, 5, 5, 6, 5, 7, 5,
    0, 6, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 7, 6, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7, 7,
  ], [
    7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 0, 0, 7, 1, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1, 1, 1, 0, 1,
    7, 2, 6, 2, 5, 2, 4, 2, 3, 2, 2, 2, 1, 2, 0, 2, 7, 3, 6, 3, 5, 3, 4, 3, 3, 3, 2, 3, 1, 3, 0, 3,
    7, 4, 6, 4, 5, 4, 4, 4, 3, 4, 2, 4, 1, 4, 0, 4, 7, 5, 6, 5, 5, 5, 4, 5, 3, 5, 2, 5, 1, 5, 0, 5,
    7, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 6, 0, 6, 7, 7, 6, 7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0, 7,
  ], [
    0, 0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 1, 7, 1, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1, 1, 1, 0,
    2, 0, 2, 1, 2, 2, 2, 3, 2, 4, 2, 5, 2, 6, 2, 7, 3, 7, 3, 6, 3, 5, 3, 4, 3, 3, 3, 2, 3, 1, 3, 0,
    4, 0, 4, 1, 4, 2, 4, 3, 4, 4, 4, 5, 4, 6, 4, 7, 5, 7, 5, 6, 5, 5, 5, 4, 5, 3, 5, 2, 5, 1, 5, 0,
    6, 0, 6, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 7, 7, 7, 7, 6, 7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0,
  ], [
    0, 7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 0, 1, 0, 1, 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7,
    2, 7, 2, 6, 2, 5, 2, 4, 2, 3, 2, 2, 2, 1, 2, 0, 3, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3, 5, 3, 6, 3, 7,
    4, 7, 4, 6, 4, 5, 4, 4, 4, 3, 4, 2, 4, 1, 4, 0, 5, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5, 5, 5, 6, 5, 7,
    6, 7, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 6, 0, 7, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7,
  ], [
    0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 7, 1, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1, 1, 1, 0, 1,
    0, 2, 1, 2, 2, 2, 3, 2, 4, 2, 5, 2, 6, 2, 7, 2, 7, 3, 6, 3, 5, 3, 4, 3, 3, 3, 2, 3, 1, 3, 0, 3,
    0, 4, 1, 4, 2, 4, 3, 4, 4, 4, 5, 4, 6, 4, 7, 4, 7, 5, 6, 5, 5, 5, 4, 5, 3, 5, 2, 5, 1, 5, 0, 5,
    0, 6, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 7, 6, 7, 7, 6, 7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0, 7,
  ], [
    7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 0, 0, 0, 1, 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1,
    7, 2, 6, 2, 5, 2, 4, 2, 3, 2, 2, 2, 1, 2, 0, 2, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3, 5, 3, 6, 3, 7, 3,
    7, 4, 6, 4, 5, 4, 4, 4, 3, 4, 2, 4, 1, 4, 0, 4, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5, 5, 5, 6, 5, 7, 5,
    7, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 6, 0, 6, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7, 7,
  ], [
    0, 0, 0, 1, 1, 0, 2, 0, 1, 1, 0, 2, 0, 3, 1, 2, 2, 1, 3, 0, 4, 0, 3, 1, 2, 2, 1, 3, 0, 4, 0, 5,
    1, 4, 2, 3, 3, 2, 4, 1, 5, 0, 6, 0, 5, 1, 4, 2, 3, 3, 2, 4, 1, 5, 0, 6, 0, 7, 1, 6, 2, 5, 3, 4,
    4, 3, 5, 2, 6, 1, 7, 0, 7, 1, 6, 2, 5, 3, 4, 4, 3, 5, 2, 6, 1, 7, 2, 7, 3, 6, 4, 5, 5, 4, 6, 3,
    7, 2, 7, 3, 6, 4, 5, 5, 4, 6, 3, 7, 4, 7, 5, 6, 6, 5, 7, 4, 7, 5, 6, 6, 5, 7, 6, 7, 7, 6, 7, 7,
  ], [
    0, 7, 0, 6, 1, 7, 2, 7, 1, 6, 0, 5, 0, 4, 1, 5, 2, 6, 3, 7, 4, 7, 3, 6, 2, 5, 1, 4, 0, 3, 0, 2,
    1, 3, 2, 4, 3, 5, 4, 6, 5, 7, 6, 7, 5, 6, 4, 5, 3, 4, 2, 3, 1, 2, 0, 1, 0, 0, 1, 1, 2, 2, 3, 3,
    4, 4, 5, 5, 6, 6, 7, 7, 7, 6, 6, 5, 5, 4, 4, 3, 3, 2, 2, 1, 1, 0, 2, 0, 3, 1, 4, 2, 5, 3, 6, 4,
    7, 5, 7, 4, 6, 3, 5, 2, 4, 1, 3, 0, 4, 0, 5, 1, 6, 2, 7, 3, 7, 2, 6, 1, 5, 0, 6, 0, 7, 1, 7, 0,
  ], [
    0, 0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7, 7, 7, 6,
    7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 1, 1, 1, 2, 1, 3, 1, 4,
    1, 5, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1,
    2, 2, 2, 3, 2, 4, 2, 5, 3, 5, 4, 5, 5, 5, 5, 4, 5, 3, 5, 2, 4, 2, 3, 2, 3, 3, 3, 4, 4, 4, 4, 3,
  ], [
    0, 0, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 7, 7, 7, 6, 7, 5, 7, 4, 7, 3, 7, 2, 7, 1, 7, 0,
    1, 0, 1, 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 6, 7, 6, 6, 6, 5, 6, 4, 6, 3, 6, 2, 6, 1, 6, 0,
    2, 0, 2, 1, 2, 2, 2, 3, 2, 4, 2, 5, 2, 6, 2, 7, 5, 7, 5, 6, 5, 5, 5, 4, 5, 3, 5, 2, 5, 1, 5, 0,
    3, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3, 5, 3, 6, 3, 7, 4, 7, 4, 6, 4, 5, 4, 4, 4, 3, 4, 2, 4, 1, 4, 0,
  ], [
    7, 0, 6, 0, 5, 0, 4, 0, 3, 0, 2, 0, 1, 0, 0, 0, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7, 5, 7, 6, 7, 7, 7,
    7, 1, 6, 1, 5, 1, 4, 1, 3, 1, 2, 1, 1, 1, 0, 1, 0, 6, 1, 6, 2, 6, 3, 6, 4, 6, 5, 6, 6, 6, 7, 6,
    7, 2, 6, 2, 5, 2, 4, 2, 3, 2, 2, 2, 1, 2, 0, 2, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5, 5, 5, 6, 5, 7, 5,
    7, 3, 6, 3, 5, 3, 4, 3, 3, 3, 2, 3, 1, 3, 0, 3, 0, 4, 1, 4, 2, 4, 3, 4, 4, 4, 5, 4, 6, 4, 7, 4,
  ], [
    0, 0, 0, 7, 7, 7, 7, 0, 1, 0, 0, 1, 0, 6, 1, 7, 6, 7, 7, 6, 7, 1, 6, 0, 2, 0, 1, 1, 0, 2, 0, 5,
    1, 6, 2, 7, 5, 7, 6, 6, 7, 5, 7, 2, 6, 1, 5, 0, 3, 0, 2, 1, 1, 2, 0, 3, 0, 4, 1, 5, 2, 6, 3, 7,
    4, 7, 5, 6, 6, 5, 7, 4, 7, 3, 6, 2, 5, 1, 4, 0, 3, 1, 2, 2, 1, 3, 1, 4, 2, 5, 3, 6, 4, 6, 5, 5,
    6, 4, 6, 3, 5, 2, 4, 1, 3, 2, 2, 3, 2, 4, 3, 5, 4, 5, 5, 4, 5, 3, 4, 2, 3, 3, 3, 4, 4, 4, 4, 3,
  ], [
    0, 0, 0, 2, 0, 4, 0, 6, 1, 1, 1, 3, 1, 5, 1, 7, 2, 0, 2, 2, 2, 4, 2, 6, 3, 1, 3, 3, 3, 5, 3, 7,
    4, 0, 4, 2, 4, 4, 4, 6, 5, 1, 5, 3, 5, 5, 5, 7, 6, 0, 6, 2, 6, 4, 6, 6, 7, 1, 7, 3, 7, 5, 7, 7,
    0, 1, 0, 3, 0, 5, 0, 7, 1, 0, 1, 2, 1, 4, 1, 6, 2, 1, 2, 3, 2, 5, 2, 7, 3, 0, 3, 2, 3, 4, 3, 6,
    4, 1, 4, 3, 4, 5, 4, 7, 5, 0, 5, 2, 5, 4, 5, 6, 6, 1, 6, 3, 6, 5, 6, 7, 7, 0, 7, 2, 7, 4, 7, 6,
  ], [
    0, 0, 2, 0, 4, 0, 6, 0, 1, 1, 3, 1, 5, 1, 7, 1, 0, 2, 2, 2, 4, 2, 6, 2, 1, 3, 3, 3, 5, 3, 7, 3,
    0, 4, 2, 4, 4, 4, 6, 4, 1, 5, 3, 5, 5, 5, 7, 5, 0, 6, 2, 6, 4, 6, 6, 6, 1, 7, 3, 7, 5, 7, 7, 7,
    1, 0, 3, 0, 5, 0, 7, 0, 0, 1, 2, 1, 4, 1, 6, 1, 1, 2, 3, 2, 5, 2, 7, 2, 0, 3, 2, 3, 4, 3, 6, 3,
    1, 4, 3, 4, 5, 4, 7, 4, 0, 5, 2, 5, 4, 5, 6, 5, 1, 6, 3, 6, 5, 6, 7, 6, 0, 7, 2, 7, 4, 7, 6, 7,
  ]
];
