use std::collections::VecDeque;
use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::input::util::imaadpcm::*;
use crate::input::util::lzss::lz_copy;
use super::super::*;

const MVI1WIDTH:  usize = 320;
const MVI1HEIGHT: usize = 200;
const MVI1ABLK: usize = 400;

#[derive(Default)]
struct AudioControl {
    aqueue:     VecDeque<Vec<u8>>,
    num_abufs:  usize,
    agroup:     u8,
    prebuf:     u8,
    local_fno:  u32,
    start:      bool,
    audio:      bool,
}

impl AudioControl {
    fn new() -> Self { Self::default() }
    fn reset(&mut self) {
        self.start = true;
        self.local_fno = 0;
    }
    fn update(&mut self) {
        self.local_fno += 1;
        self.start = false;
        self.audio = true;
    }
    fn has_audio(&self, flush: bool) -> bool { (self.audio || flush) && !self.aqueue.is_empty() }
    fn get_buf(&mut self) -> Vec<u8> {
        self.audio = false;
        self.aqueue.pop_front().unwrap()
    }
    fn read_prebuffered_audio(&mut self, br: &mut dyn ByteIO) -> DecoderResult<()> {
        if self.agroup > 0 && (self.local_fno % u32::from(self.agroup)) == 0 {
            let num_abufs = if self.start {
                    usize::from(self.prebuf) * usize::from(self.agroup)
                } else { usize::from(self.agroup) };
            for _ in 0..num_abufs {
                self.read_frame_audio(br)?;
            }
        }
        Ok(())
    }
    fn read_frame_audio(&mut self, br: &mut dyn ByteIO) -> DecoderResult<()> {
        let mut abuf = vec![0; MVI1ABLK * self.num_abufs];
        br.read_buf(&mut abuf)?;
        self.aqueue.push_back(abuf);
        Ok(())
    }
}

struct MVI1Decoder {
    fr:         FileReader<BufReader<File>>,
    frm:        Vec<u8>,
    pfrm:       Vec<u8>,
    pal:        [u8; 768],
    frm_sizes:  Vec<u32>,
    version:    u16,
    data:       Vec<u8>,
    actl:       AudioControl,
}

fn read_varint(br: &mut dyn ByteIO, maxshift: u8) -> DecoderResult<u32> {
    let mut val = 0;
    let mut shift = 0;
    loop {
        validate!(shift <= maxshift);
        let b = br.read_byte()?;
        val |= u32::from(b & 0x7F) << shift;
        if (b & 0x80) == 0 {
            break;
        }
        shift += 7;
    }
    Ok(val)
}

fn lz_unpack(br: &mut dyn ByteIO, dst: &mut [u8]) -> DecoderResult<()> {
    let mut pos = 0;
    while pos < dst.len() {
        let op = br.read_byte()?;
        match op {
            0x00 => {
                let len = usize::from(br.read_u16le()?);
                validate!(pos + len <= dst.len());
                pos += len;
            },
            0x01..=0x7F => {
                let len = usize::from(op);
                validate!(pos + len <= dst.len());
                pos += len;
            },
            0x80..=0x9F => {
                let len = usize::from(op & 0x1F) + 1;
                validate!(pos + len <= dst.len());
                br.read_buf(&mut dst[pos..][..len])?;
                pos += len;
            },
            0xA0..=0xBF => {
                let len = usize::from(op & 0x1F) + 2;
                validate!(pos + len <= dst.len());
                let clr = br.read_byte()?;
                for el in dst[pos..][..len].iter_mut() {
                    *el = clr;
                }
                pos += len;
            },
            _ => {
                let len = usize::from(op & 0x3F) + 2;
                validate!(pos + len <= dst.len());
                let offs = read_varint(br, 14)? as usize;
                validate!(offs > 0);
                lz_copy(dst, pos, offs, len);
                pos += len;
            },
        }
    }
    Ok(())
}

impl InputSource for MVI1Decoder {
    // theoretically version 0x302 may contain audio but no such sample known
    fn get_num_streams(&self) -> usize { if self.version == 0x310 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  MVI1WIDTH,
                height: MVI1HEIGHT,
                bpp:    8,
                tb_num: 60,
                tb_den: 1000,
            }),
            1 if self.version == 0x310 => StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::U8,
                channels:    2,
            }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.actl.has_audio(self.frm_sizes.is_empty()) {
            let aud = self.actl.get_buf();
            return Ok((1, Frame::AudioU8(aud)));
        }
        if self.frm_sizes.is_empty() {
            let nframes = usize::from(self.fr.read_u16le().map_err(|_| DecoderError::EOF)?);
            validate!(nframes > 0);
            self.actl.num_abufs = usize::from(self.fr.read_u16le()?);
            validate!(self.actl.num_abufs > 0);
            if self.version == 0x310 {
                self.actl.agroup = self.fr.read_byte()?;
                self.actl.prebuf = self.fr.read_byte()?;
                if self.actl.agroup > 0 {
                    validate!(self.actl.prebuf > 0);
                }
            } else {
                self.actl.agroup = 0;
            }
            let toc_size = self.fr.read_u16le()?;
            let toc_end = self.fr.tell() + u64::from(toc_size);
            for _ in 0..nframes {
                let size = read_varint(&mut self.fr, 21)?;
                validate!((1..=1048576).contains(&size));
                self.frm_sizes.push(size);
            }
            validate!(self.fr.tell() == toc_end);

            self.actl.reset();
        }

        let frm_size = self.frm_sizes.remove(0) as usize;
        self.data.resize(frm_size, 0);
        self.fr.read_buf(&mut self.data)?;

        std::mem::swap(&mut self.frm, &mut self.pfrm);

        let mut br = MemoryReader::new_read(&self.data);
        self.actl.read_prebuffered_audio(&mut br)?;
        let flags = br.read_byte()?;
        if (flags & 1) != 0 {
            validate!(self.actl.agroup == 0);
            self.actl.read_frame_audio(&mut br)?;
        }
        if (flags & 2) == 0 {
            self.frm.copy_from_slice(&self.pfrm);
        } else {
            // motion-compensated frame, no samples are known
            //
            // data format: RLE-compressed MV table, 0x600 bytes long
            // RLE format: 0x00..=0x7F -> copy 1..128 bytes, 0x80..=0xFF -> run of 2..129
            // MV table provides offsets for source of each 10x10 tile,
            // table data is stored de-interleaved i.e. low 768 bytes first
            // then top bytes of each offset
            return Err(DecoderError::NotImplemented);
        }
        if (flags & 4) != 0 {
            lz_unpack(&mut br, &mut self.frm).map_err(|_| DecoderError::InvalidData)?;
        }
        if (flags & 8) != 0 {
            let mut palbuf = [0; 768];
            lz_unpack(&mut br, &mut palbuf).map_err(|_| DecoderError::InvalidData)?;
            let (sr, ss) = palbuf.split_at(0x100);
            let (sg, sb) = ss.split_at(0x100);
            for (clr, (&r, (&g, &b))) in self.pal.chunks_exact_mut(3)
                    .zip(sr.iter().zip(sg.iter().zip(sb.iter()))) {
                clr[0] = (r << 2) | (r >> 4);
                clr[1] = (g << 2) | (g >> 4);
                clr[2] = (b << 2) | (b >> 4);
            }
        }
        self.actl.update();

        Ok((0, Frame::VideoPal(self.frm[..MVI1WIDTH * MVI1HEIGHT].to_vec(), self.pal)))
    }
}

pub fn open_mvi1(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut tag = [0; 6];
    fr.read_buf(&mut tag)?;
    validate!(&tag == b"VR6MVI");
    let version = fr.read_u16le()?;
    validate!(version == 0x302 || version == 0x310);

    Ok(Box::new(MVI1Decoder {
        fr,
        frm: vec![0; MVI1WIDTH * 240],
        pfrm: vec![0; MVI1WIDTH * 240],
        pal: [0; 768],
        version,
        frm_sizes: Vec::new(),
        data: Vec::new(),
        actl: AudioControl::new(),
    }))
}

struct YUV2RGB {
    tab:    [u16; 32768],
}

impl YUV2RGB {
    fn new() -> Self {
        Self { tab: std::array::from_fn(Self::idx2rgb) }
    }
    fn idx2rgb(idx: usize) -> u16 {
        let cu = ((idx >> 10) as i32 - 0xF) * 0xF / 0x18;
        let y = ((idx >> 5) & 0x1F) as i32;
        let cv = ((idx & 0x1F) as i32 - 0xF) * 0xF / 0x2C;

        let r = (y + (cu * 1140 + cv *  650) / 1000).max(0).min(0x1F) as u16;
        let g = (y - (cu *  324 + cv *  677) / 1000).max(0).min(0x1F) as u16;
        let b = (y - (cu * 1317 - cv * 1780) / 1000).max(0).min(0x1F) as u16;

        (r << 10) | (g << 5) | b
    }
}

const NUM_TILES: usize = 4096;

struct MVI2Decoder {
    fr:         FileReader<BufReader<File>>,
    frm:        Vec<u16>,
    tiles:      Vec<[u8; 16]>,
    idx_buf:    Vec<u8>,
    delay:      u16,
    tile_w:     usize,
    tile_h:     usize,
    tile_idx:   usize,
    frm_sizes:  Vec<u32>,
    frameno:    usize,
    abuffers:   usize,
    abuf:       Vec<i16>,
    ablk_len:   usize,
    yuv2rgb:    YUV2RGB,
    ima:        [IMAState; 2],
}

fn render_tile(bits: &mut BitReader, dst: &mut [u16], stride: usize, tiles: &[[u8; 16]], yuv2rgb: &[u16; 32768]) -> DecoderResult<()> {
    for strip in dst.chunks_mut(stride * 4) {
        for x in (0..64).step_by(4) {
            let idx = bits.read(12)? as usize;
            for (dline, sline) in strip[x..].chunks_mut(stride).zip(tiles[idx].chunks_exact(4)) {
                for (pix, &s) in dline[..4].iter_mut().zip(sline.iter()) {
                    *pix = u16::from(s) << 5;
                }
            }
        }
    }
    for &shift in [0, 10].iter() {
        for strip in dst.chunks_mut(stride * 8) {
            for x in (0..32).step_by(4) {
                let idx = bits.read(12)? as usize;
                for (dline, sline) in strip[x * 2..].chunks_mut(stride * 2).zip(tiles[idx].chunks_exact(4)) {
                    for (pair, &s) in dline[..8].chunks_exact_mut(2).zip(sline.iter()) {
                        pair[0] |= u16::from(s) << shift;
                        pair[1] |= u16::from(s) << shift;
                    }
                    for (pair, &s) in dline[stride..][..8].chunks_exact_mut(2).zip(sline.iter()) {
                        pair[0] |= u16::from(s) << shift;
                        pair[1] |= u16::from(s) << shift;
                    }
                }
            }
        }
    }

    for line in dst.chunks_mut(stride) {
        for pix in line[..64].iter_mut() {
            *pix = yuv2rgb[usize::from(*pix)];
        }
    }
    Ok(())
}

impl InputSource for MVI2Decoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.tile_w * 64,
                height: self.tile_h * 64,
                bpp:    15,
                tb_num: u32::from(self.delay),
                tb_den: 1000,
            }),
            1 => StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::S16,
                channels:    2,
            }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.abuf.len() >= self.ablk_len {
            let mut audio = vec![0; self.ablk_len];
            audio.copy_from_slice(&self.abuf[..self.ablk_len]);
            self.abuf.drain(..self.ablk_len);
            return Ok((1, Frame::AudioS16(audio)));
        }

        if self.frameno >= self.frm_sizes.len() {
            return Err(DecoderError::EOF);
        }

        let br = &mut self.fr;
        let end = br.tell() + (self.frm_sizes[self.frameno] as u64);

        let _flags = br.read_byte()?;

        let audio_len = if self.frameno == 0 {
                usize::from(self.delay) * (self.abuffers + 1) * 2205 / 100
            } else {
                usize::from(self.delay) * (self.abuffers + self.frameno + 1) * 2205 / 100 -
                usize::from(self.delay) * (self.abuffers + self.frameno) * 2205 / 100
            };
        self.frameno += 1;

        for _ in 0..audio_len {
            let b = br.read_byte()?;
            let sample0 = self.ima[0].expand_sample(b & 0xF);
            let sample1 = self.ima[1].expand_sample(b >> 4);
            self.abuf.push(sample0);
            self.abuf.push(sample1);
        }

        let num_tiles = usize::from(br.read_u16le()?);
        validate!(num_tiles <= 4096);
        let mut bbuf = [0; 10];
        for _ in 0..num_tiles {
            br.read_buf(&mut bbuf)?;
            let mut bits = BitReader::new(&bbuf, BitReaderMode::LE);
            for dst in self.tiles[self.tile_idx].iter_mut() {
                *dst = bits.read(5)? as u8;
            }
            self.tile_idx = (self.tile_idx + 1) & (NUM_TILES - 1);
        }

        br.read_buf(&mut self.idx_buf)?;
        validate!(br.tell() <= end);
        let mut bits = BitReader::new(&self.idx_buf, BitReaderMode::LE);
        let stride = self.tile_w * 64;
        for strip in self.frm.chunks_exact_mut(stride * 64) {
            for tx in 0..self.tile_w {
                render_tile(&mut bits, &mut strip[tx * 64..], stride, &self.tiles, &self.yuv2rgb.tab)?;
            }
        }
        br.seek(SeekFrom::Start(end))?;

        Ok((0, Frame::VideoRGB16(self.frm.clone())))
    }
}

pub fn open_mvi2(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"MVI2");
    let version = fr.read_u16le()?;
    validate!(version == 0x101);
    let nframes = usize::from(fr.read_u16le()?);
    validate!(nframes > 0);
    let delay = fr.read_u16le()?;
    validate!((10..=1000).contains(&delay));
    let tile_w = usize::from(fr.read_byte()?);
    let tile_h = usize::from(fr.read_byte()?);
    let mut params = [0; 6];
    fr.read_buf(&mut params)?;
    validate!(params == [0x05, 0x05, 0x05, 0x0C, 0x02, 0x02]);
    let abuffers = usize::from(fr.read_byte()?);
    let _flags = fr.read_u32le()?;
    let mut frm_sizes = Vec::with_capacity(nframes);
    let afrm_len = usize::from(delay) * 2205 / 100;
    for _ in 0..nframes {
        let fsize = fr.read_u32le()?;
        validate!(fsize as usize > afrm_len + 5);
        frm_sizes.push(fsize);
    }

    Ok(Box::new(MVI2Decoder {
        fr,
        yuv2rgb: YUV2RGB::new(),
        tile_w, tile_h, delay,
        frm: vec![0; tile_w * 64 * tile_h * 64],
        tiles: vec![[0; 16]; NUM_TILES],
        tile_idx: 0,
        idx_buf: vec![0; (tile_w * 16 * tile_h * 16 * 3 / 2 * 12 + 4) / 8],
        frm_sizes,
        frameno: 0,
        abuffers,
        abuf: Vec::new(),
        ablk_len: (afrm_len + 1) & !1,
        ima: [IMAState::new(), IMAState::new()],
    }))
}
