use crate::input::{DecoderResult,DecoderError};
use crate::io::bitreader::*;
use crate::io::byteio::*;
use crate::input::util::jpeg::{idct, ZIGZAG};

pub struct HNM6Dec {
    blocks:     [[i16; 64]; 3],
    width:      usize,
    height:     usize,
    stride:     usize,
    ymat:       [i16; 64],
    cmat:       [i16; 64],
    q:          u8,
}

impl HNM6Dec {
    pub fn new(width: usize, height: usize) -> Self {
        Self {
            width, height,
            stride: width * 3,
            blocks: [[0; 64]; 3],
            ymat: [0; 64],
            cmat: [0; 64],
            q: 0,
        }
    }
    fn set_quality(&mut self, q: u8) {
        if self.q == q {
            return;
        }
        self.q = q;
        let scale = if q < 50 { 5000 / i32::from(q) } else { 200 - 2 * i32::from(q) };

        for (dq, &sq) in self.ymat.iter_mut().zip(LUMA_QMAT.iter()) {
            *dq = ((i32::from(sq) * scale + 50) / 100).clamp(8, 255) as i16;
        }
        for (dq, &sq) in self.cmat.iter_mut().zip(CHROMA_QMAT.iter()) {
            *dq = ((i32::from(sq) * scale + 50) / 100).clamp(8, 255) as i16;
        }
    }
    fn decode_dct(blk: &mut [i16; 64], br: &mut DctReader) -> DecoderResult<()> {
        *blk = [0; 64];
        blk[0] = br.read_dc()?;
        let mut idx = 1;
        while idx < blk.len() {
            match br.read_nib()? {
                0 => return Ok(()),
                1 => {
                    validate!(idx + 5 < blk.len());
                    idx += 5;
                },
                2 => {
                    validate!(idx + 2 <= blk.len());
                    blk[ZIGZAG[idx + 2 - 1]] = 1;
                    idx += 2;
                },
                3 => {
                    validate!(idx + 2 <= blk.len());
                    blk[ZIGZAG[idx + 2 - 1]] = -1;
                    idx += 2;
                },
                4 => {
                    validate!(idx + 3 <= blk.len());
                    blk[ZIGZAG[idx + 3 - 1]] = 1;
                    idx += 3;
                },
                5 => {
                    validate!(idx + 3 <= blk.len());
                    blk[ZIGZAG[idx + 3 - 1]] = -1;
                    idx += 3;
                },
                6 => {
                    validate!(idx + 4 <= blk.len());
                    blk[ZIGZAG[idx + 4 - 1]] = 1;
                    idx += 4;
                },
                7 => {
                    validate!(idx + 4 <= blk.len());
                    blk[ZIGZAG[idx + 4 - 1]] = -1;
                    idx += 4;
                },
                8 => {
                    let cnt = usize::from(br.read_nib()?);
                    let skip = (cnt >> 2) + 1;
                    let nc = (cnt & 3) + 2;
                    validate!(idx + skip + nc <= blk.len());
                    idx += skip;
                    for _ in 0..nc/2 {
                        let (c0, c1) = br.read_22()?;
                        blk[ZIGZAG[idx]] = c0;
                        blk[ZIGZAG[idx + 1]] = c1;
                        idx += 2;
                    }
                    if (nc & 1) != 0 {
                        let (c0, _c1) = br.read_22()?;
                        blk[ZIGZAG[idx]] = c0;
                        idx += 1;
                    }
                },
                9 => {
                    let cnt = usize::from(br.read_nib()?);
                    let skip = (cnt >> 2) + 1;
                    let nc = (cnt & 3) + 1;
                    validate!(idx + skip + nc <= blk.len());
                    idx += skip;
                    for _ in 0..nc {
                        blk[ZIGZAG[idx]] = br.read_4()?;
                        idx += 1;
                    }
                },
                10 => {
                    validate!(idx + 2 <= blk.len());
                    let (c0, c1) = br.read_22()?;
                    blk[ZIGZAG[idx]] = c0;
                    blk[ZIGZAG[idx + 1]] = c1;
                    idx += 2;
                },
                11 => {
                    blk[ZIGZAG[idx]] = br.read_4()?;
                    idx += 1;
                },
                12 => {
                    validate!(idx + 2 <= blk.len());
                    blk[ZIGZAG[idx]] = br.read_4()?;
                    blk[ZIGZAG[idx + 1]] = br.read_4()?;
                    idx += 2;
                },
                13 => {
                    validate!(idx + 3 <= blk.len());
                    blk[ZIGZAG[idx]] = br.read_4()?;
                    blk[ZIGZAG[idx + 1]] = br.read_4()?;
                    blk[ZIGZAG[idx + 2]] = br.read_4()?;
                    idx += 3;
                },
                14 => {
                    idx += 1;
                },
                15 => {
                    blk[ZIGZAG[idx]] = br.read_dc()?;
                    idx += 1;
                },
                _ => unreachable!(),
            }
        }
        Ok(())
    }
    #[allow(clippy::too_many_arguments)]
    fn decode_block(&mut self, frame: &mut [u8], pframe: &[u8], x: usize, y: usize, bmode: BlockMode, bits: &mut BitReader, dct: &mut DctReader, long_mvs: &mut MemoryReader, short_mvs: &mut ShortMVReader) -> DecoderResult<()> {
        let op = bmode.read_op(bits)?;
        match op {
            BlockOp::DCT => {
                let qmats = [&self.ymat, &self.cmat, &self.cmat];
                for (blk, qmat) in self.blocks.iter_mut().zip(qmats.iter()) {
                    Self::decode_dct(blk, dct)?;
                    for (coef, &q) in blk.iter_mut().zip(qmat.iter()) {
                        *coef *= q;
                    }
                    idct(blk);
                }
                for ((dline, yline), (uline, vline)) in frame.chunks_exact_mut(self.stride)
                        .skip(y).take(8).zip(self.blocks[0].chunks_exact(8))
                        .zip(self.blocks[1].chunks_exact(8).zip(self.blocks[2].chunks_exact(8))) {
                    for ((dst, &y), (&u, &v)) in dline[x * 3..].chunks_exact_mut(3)
                            .zip(yline.iter()).zip(uline.iter().zip(vline.iter())) {
                        // an approximation of real formula
                        let y = y + 128;
                        let cr = (16 * v + 8) / 10;
                        let cb = u / 3;

                        dst[0] = (y + cr).clamp(0, 255) as u8;
                        dst[1] = (y - cr / 2 - cb).clamp(0, 255) as u8;
                        dst[2] = (y + cb).clamp(0, 255) as u8;
                    }
                }
            },
            BlockOp::SplitH => {
                let subbm = bmode.split_h();
                self.decode_block(frame, pframe, x, y, subbm, bits, dct, long_mvs, short_mvs)?;
                self.decode_block(frame, pframe, x, y + usize::from(subbm.height), subbm, bits, dct, long_mvs, short_mvs)?;
            },
            BlockOp::SplitV => {
                let subbm = bmode.split_v();
                self.decode_block(frame, pframe, x, y, subbm, bits, dct, long_mvs, short_mvs)?;
                self.decode_block(frame, pframe, x + usize::from(subbm.width), y, subbm, bits, dct, long_mvs, short_mvs)?;
            },
            BlockOp::Split4 => {
                let subbm = bmode.split_4();
                self.decode_block(frame, pframe, x, y, subbm, bits, dct, long_mvs, short_mvs)?;
                self.decode_block(frame, pframe, x + usize::from(subbm.width), y, subbm, bits, dct, long_mvs, short_mvs)?;
                self.decode_block(frame, pframe, x, y + usize::from(subbm.height), subbm, bits, dct, long_mvs, short_mvs)?;
                self.decode_block(frame, pframe, x + usize::from(subbm.width), y + usize::from(subbm.height), subbm, bits, dct, long_mvs, short_mvs)?;
            },
            BlockOp::Motion => {
                let mword = long_mvs.read_u16le()?;
                let (mut dx, mut dy, cur, copy_mode) = bmode.translate_mv(mword, bits)?;
                dx -= (x & 7) as i16;
                dy -= (y & 7) as i16;
                if bmode.width >= 4 && bmode.height >= 4 {
                    if dx + (x as i16) < 0 {
                        dx += self.width as i16;
                    } else if dx + (x as i16) >= self.width as i16 {
                        dx -= self.width as i16;
                    }
                }
                let mv = isize::from(dx) * 3 + isize::from(dy) * (self.stride as isize);
                hnm6_copy_block(frame, pframe, x, y, self.stride, usize::from(bmode.width), usize::from(bmode.height), copy_mode, cur, mv)?;
            },
            BlockOp::ShortM => {
                let smv = short_mvs.read()?;
                let (dx, dy, use_cur, cmode) = if !bmode.warp {
                        let (dx, dy) = short_mv_code_uncoil(smv);
                        (dx, dy, false, 0)
                    } else {
                        let (dx, dy, cmode) = warp_short_code(smv);
                        (dx - ((x & 7) as i8), dy - ((y & 7) as i8), true, cmode)
                    };
                let mv = isize::from(dx) * 3 + isize::from(dy) * (self.stride as isize);
                hnm6_copy_block(frame, pframe, x, y, self.stride, usize::from(bmode.width), usize::from(bmode.height), cmode, use_cur, mv)?;
            },
            BlockOp::Skip => {
                let width = usize::from(bmode.width);
                for (dline, sline) in frame.chunks_exact_mut(self.stride)
                        .zip(pframe.chunks_exact(self.stride))
                        .skip(y).take(usize::from(bmode.height)) {
                    dline[x * 3..][..width * 3].copy_from_slice(&sline[x * 3..][..width * 3]);
                }
            },
        }
        Ok(())
    }
    pub fn decode(&mut self, src: &[u8], frame: &mut [u8], pframe: &[u8], warp: bool) -> DecoderResult<()> {
        validate!(src.len() > 24);
        let quality  = read_u32le(src).unwrap_or_default() as i32;
        validate!((-100..=100).contains(&quality) && quality != 0);
        let bits_off = read_u32le(&src[4..]).unwrap_or_default() as usize;
        let mot_off  = read_u32le(&src[8..]).unwrap_or_default() as usize;
        let smot_off = read_u32le(&src[12..]).unwrap_or_default() as usize;
        let dct_off  = read_u32le(&src[16..]).unwrap_or_default() as usize;
        let dct_end  = read_u32le(&src[20..]).unwrap_or_default() as usize;
        validate!(bits_off <= src.len());
        validate!(mot_off <= src.len());
        validate!(smot_off <= src.len());
        validate!(dct_off <= dct_end && dct_end <= src.len());
        let mut bits = BitReader::new(&src[bits_off..], BitReaderMode::LE32MSB);
        let mut dct = DctReader::new(&src[dct_off..dct_end]);
        let mut long_mvs = MemoryReader::new_read(&src[mot_off..]);
        let mut short_mvs = ShortMVReader::new(&src[smot_off..]);

        let is_kf = quality < 0;
        self.set_quality(quality.unsigned_abs() as u8);
        let bmode = BlockMode::new(is_kf, warp);
        for y in (0..self.height).step_by(8) {
            for x in (0..self.width).step_by(8) {
                self.decode_block(frame, pframe, x, y, bmode, &mut bits, &mut dct, &mut long_mvs, &mut short_mvs)?
            }
        }

        Ok(())
    }
}

fn warp_short_code(val: u16) -> (i8, i8, u8) {
    let mv_idx = ((val & 1) << 8) | (val & 0xF0) | (val >> 8);
    let cmode = (val >> 1) & 7;

    if mv_idx < 12 * 8 {
        (-2 - ((mv_idx % 12) as i8), 6 - ((mv_idx / 12) as i8), cmode as u8)
    } else {
        let     dx = 19 - (((mv_idx - 12 * 8) % 32) as i8);
        let mut dy = -2 - (((mv_idx - 12 * 8) / 32) as i8);
        if dy <= -8 {
            dy -= 1;
        }
        (dx, dy, cmode as u8)
    }
}

// convert position on spiral starting at (1,0) into planar co-ordinates
fn short_mv_code_uncoil(val: u16) -> (i8, i8) {
    let mut dx = 1;
    let mut dy = 0;
    let mut lim = 4;
    let mut llim = 0;
    let mut edge = 2;
    while val >= lim {
        llim = lim;
        edge += 2;
        lim = edge * edge;
        dx += 1;
        dy += 1;
    }
    edge -= 1;
    let dir = (val - llim) / edge;
    let pos = (val - llim) % edge;
    match dir {
        0 => { // right-up
            dy -= pos as i8;
        },
        1 => {
            dx -= (pos + 1) as i8;
            dy -= (edge - 1) as i8;
        },
        2 => {
            dx -= edge as i8;
            dy += (pos + 2) as i8;
            dy -= edge as i8;
        },
        3 => {
            dx += (pos + 1) as i8;
            dx -= edge as i8;
            dy += 1;
        },
        _ => unreachable!()
    }
    (dx, dy)
}

#[derive(Clone,Copy)]
struct BlockMode {
    keyframe:   bool,
    warp:       bool,
    width:      u8,
    height:     u8,
}

impl BlockMode {
    fn new(keyframe: bool, warp: bool) -> Self {
        Self { keyframe, warp, width: 8, height: 8 }
    }
    fn split_h(mut self) -> Self {
        self.height >>= 1;
        self
    }
    fn split_v(mut self) -> Self {
        self.width >>= 1;
        self
    }
    fn split_4(mut self) -> Self {
        self.width  >>= 1;
        self.height >>= 1;
        self
    }
    fn read_op(self, br: &mut BitReader) -> DecoderResult<BlockOp> {
        if !self.warp {
            let kf_idx = if self.keyframe { 0 } else { 1 };
            match (self.width, self.height) {
                (8, 8) => {
                    let mode = br.read_h6cb(&H6CB_8X8[kf_idx])?;
                    if !self.keyframe && mode == BlockOp::Split4 {
                        let bit = br.read_bool()?;
                        validate!(!bit);
                    }
                    Ok(mode)
                },
                (8, 4) => br.read_h6cb(&H6CB_8X4[kf_idx]),
                (4, 8) => br.read_h6cb(&H6CB_4X8[kf_idx]),
                (4, 4) => br.read_h6cb(&H6CB_4X4[kf_idx]),
                (4, 2) => br.read_h6cb(&H6CB_4X2[kf_idx]),
                (2, 4) => br.read_h6cb(&H6CB_2X4[kf_idx]),
                (2, 2) if self.keyframe => Ok(BlockOp::Motion),
                (2, 2) => br.read_h6cb(&H6CB_2X2),
                _ => unreachable!()
            }
        } else {
            match (self.width, self.height) {
                (8, 8) => br.read_h6cb(&H6CB_8X8_WARP),
                (4, 4) => br.read_h6cb(&H6CB_4X4_WARP),
                (2, 2) => Ok(BlockOp::ShortM),
                _ => unreachable!()
            }
        }
    }
    fn translate_mv(self, mv: u16, bits: &mut BitReader) -> DecoderResult<(i16, i16, bool, u8)> {
        let large_blk = self.width >= 4 && self.height >= 4;
        let use_cur = (mv & 0x8000) != 0;
        let smode = ((mv >> 12) & 7) as u8;
        if self.warp || (self.keyframe && large_blk) {
            let mode = bits.read(2)? * 2 + u32::from(mv >> 15);
            let ybias = if self.width == 8 && self.height == 8 { 0 } else { 4 };
            let dx = ((mv >> 7) & 0xFF) as i16;
            let dy = (mv & 0x7F) as i16;
            Ok((128 - dx, ybias - dy, true, mode as u8))
        } else if self.keyframe || (!large_blk && use_cur) {
            let dx = (mv & 0x7F) as i16;
            let dy = ((mv >> 7) & 0x1F) as i16;
            Ok((63 - dx, 6 - dy, true, smode))
        } else if large_blk {
            let mode = bits.read(3)? as u8;
            let ybias = if !use_cur { 64 } else if self.width == 8 && self.height == 8 { 0 } else { 4 };
            let dx = ((mv >> 7) & 0xFF) as i16;
            let dy = (mv & 0x7F) as i16;
            Ok((128 - dx, ybias - dy, use_cur, mode))
        } else {
            let dx = (mv & 0x3F) as i16;
            let dy = ((mv >> 6) & 0x3F) as i16;
            Ok((31 - dx, 31 - dy, false, smode))
        }
    }
}

#[allow(clippy::too_many_arguments)]
fn hnm6_copy_block(frame: &mut [u8], pframe: &[u8], xpos: usize, ypos: usize, stride: usize, blk_w: usize, blk_h: usize, mut copy_mode: u8, copy_cur: bool, mv_offset: isize) -> DecoderResult<()> {
    let mut dst_off = xpos * 3 + ypos * stride;
    // transpose and flip+mirror transpose mode meanings are swapped
    if copy_mode == 4 || copy_mode == 7 {
        copy_mode ^= 3;
    }
    if copy_mode == 0 {
        let src_off = (dst_off as isize) + mv_offset;
        validate!(src_off >= 0);
        let mut src_off = src_off as usize;
        let src_end = src_off + blk_w * 3 + (blk_h - 1) * stride;
        validate!(src_end <= frame.len());
        if !copy_cur {
            for (dline, sline) in frame[dst_off..].chunks_mut(stride)
                    .zip(pframe[src_off..].chunks(stride)).take(blk_h) {
                dline[..blk_w * 3].copy_from_slice(&sline[..blk_w * 3]);
            }
        } else {
            for _ in 0..blk_h {
                for x in 0..(blk_w * 3) {
                    frame[dst_off + x] = frame[src_off + x];
                }
                dst_off += stride;
                src_off += stride;
            }
        }
    } else {
        let mirror    = (copy_mode & 1) != 0;
        let flip      = (copy_mode & 2) != 0;
        let transpose = (copy_mode & 4) != 0;

        let mut src_off = (dst_off as isize) + mv_offset;

        let mut sstep = 3;
        let mut sstride = stride as isize;
        if transpose {
            std::mem::swap(&mut sstep, &mut sstride);
        }
        if flip {
            sstride = -sstride;
        }
        if mirror {
            sstep = -sstep;
        }
        if flip {
            src_off -= sstride * (blk_h as isize - 1);
        }
        if mirror {
            src_off -= sstep * (blk_w as isize - 1);
        }
        for _y in 0..blk_h {
            let mut sidx = src_off;
            for x in 0..blk_w {
                validate!(sidx >= 0 && (sidx as usize) + 2 < frame.len());
                if !copy_cur {
                    frame[dst_off + x * 3..][..3].copy_from_slice(&pframe[sidx as usize..][..3]);
                } else {
                    for i in 0..3 {
                        frame[dst_off + x * 3 + i] = frame[sidx as usize + i];
                    }
                }
                sidx += sstep;
            }
            dst_off += stride;
            src_off += sstride;
        }
    }
    Ok(())
}


#[derive(Clone,Copy,Debug,PartialEq)]
enum BlockOp {
    DCT,
    SplitH,
    SplitV,
    Split4,
    Motion,
    Skip,
    ShortM
}

type H6CB = [(BlockOp, u8); 8];

trait ReadH6CB {
    fn read_h6cb(&mut self, cb: &H6CB) -> DecoderResult<BlockOp>;
}

impl<'a> ReadH6CB for BitReader<'a> {
    fn read_h6cb(&mut self, cb: &H6CB) -> DecoderResult<BlockOp> {
        let idx = self.peek(3) as usize;
        let val = cb[idx].0;
        self.skip(u32::from(cb[idx].1))?;
        Ok(val)
    }
}

struct DctReader<'a> {
    src: &'a [u8],
    pos: usize,
    nib: bool,
    val: u8,
}

impl<'a> DctReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self { src, pos: 0, nib: false, val: 0 }
    }
    fn read_dc(&mut self) -> DecoderResult<i16> {
        let n0 = self.read_nib()?;
        let n1 = self.read_nib()?;
        Ok(i16::from(((n0 << 4) | n1) as i8))
    }
    fn read_4(&mut self) -> DecoderResult<i16> {
        let val = self.read_nib()? as i8;
        Ok(i16::from(val << 4 >> 4))
    }
    fn read_22(&mut self) -> DecoderResult<(i16, i16)> {
        let val = usize::from(self.read_nib()?);
        const MAP: [(i16, i16); 16] = [
            ( 1, 1), ( 1, 2), ( 1, -2), ( 1, -1),
            ( 2, 1), ( 2, 2), ( 2, -2), ( 2, -1),
            (-2, 1), (-2, 2), (-2, -2), (-2, -1),
            (-1, 1), (-1, 2), (-1, -2), (-1, -1)
        ];
        Ok(MAP[val])
    }
    fn read_nib(&mut self) -> DecoderResult<u8> {
        if !self.nib {
            self.nib = true;
            validate!(self.pos < self.src.len());
            self.val = self.src[self.pos];
            self.pos += 1;
            Ok(self.val & 0xF)
        } else {
            self.nib = false;
            Ok(self.val >> 4)
        }
    }
}

struct ShortMVReader<'a> {
    src:    &'a [u8],
    pos:    usize,
    buf:    u16,
    cached: bool,
}

impl<'a> ShortMVReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self { src, pos: 0, buf: 0, cached: false }
    }
    fn read(&mut self) -> DecoderResult<u16> {
        if !self.cached {
            validate!(self.pos + 2 <= self.src.len());
            let val = read_u16le(&self.src[self.pos..]).unwrap_or_default();
            self.pos += 2;
            self.cached = true;
            self.buf = val >> 12;
            Ok(val & 0xFFF)
        } else {
            validate!(self.pos < self.src.len());
            let val = u16::from(self.src[self.pos]) * 16 + self.buf;
            self.pos += 1;
            self.cached = false;
            Ok(val)
        }
    }
}

static H6CB_8X8_WARP: H6CB = [
    (BlockOp::Split4, 1), (BlockOp::Split4, 1),
    (BlockOp::Split4, 1), (BlockOp::Split4, 1),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::DCT,    2), (BlockOp::DCT,    2)
];
static H6CB_4X4_WARP: H6CB = [
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Split4, 1), (BlockOp::Split4, 1),
    (BlockOp::Split4, 1), (BlockOp::Split4, 1)
];

static H6CB_8X8: [H6CB; 2] = [
  [
    (BlockOp::SplitH, 2), (BlockOp::SplitH, 2),
    (BlockOp::SplitV, 2), (BlockOp::SplitV, 2),
    (BlockOp::Split4, 2), (BlockOp::Split4, 2),
    (BlockOp::DCT,    3), (BlockOp::Motion, 3)
  ], [
    (BlockOp::ShortM, 2), (BlockOp::ShortM, 2),
    (BlockOp::Motion, 3), (BlockOp::DCT,    3),
    (BlockOp::SplitH, 3), (BlockOp::SplitV, 3),
    (BlockOp::Skip,   3), (BlockOp::Split4, 3)
  ]
];
static H6CB_4X8: [H6CB; 2] = [
  [
    (BlockOp::SplitH, 1), (BlockOp::SplitH, 1),
    (BlockOp::SplitH, 1), (BlockOp::SplitH, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1)
  ], [
    (BlockOp::ShortM, 2), (BlockOp::ShortM, 2),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::SplitH, 2), (BlockOp::SplitH, 2),
    (BlockOp::Skip,   2), (BlockOp::Skip,   2)
  ]
];
static H6CB_8X4: [H6CB; 2] = [
  [
    (BlockOp::SplitV, 1), (BlockOp::SplitV, 1),
    (BlockOp::SplitV, 1), (BlockOp::SplitV, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1)
  ], [
    (BlockOp::ShortM, 2), (BlockOp::ShortM, 2),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::SplitV, 2), (BlockOp::SplitV, 2),
    (BlockOp::Skip,   2), (BlockOp::Skip,   2)
  ]
];
static H6CB_4X4: [H6CB; 2] = [
  [
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::SplitH, 2), (BlockOp::SplitH, 2),
    (BlockOp::SplitV, 2), (BlockOp::SplitV, 2),
    (BlockOp::Split4, 2), (BlockOp::Split4, 2)
  ], [
    (BlockOp::ShortM, 2), (BlockOp::ShortM, 2),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::Skip,   3), (BlockOp::SplitH, 3),
    (BlockOp::SplitV, 3), (BlockOp::Split4, 3)
  ]
];
static H6CB_2X4: [H6CB; 2] = [
  [
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::SplitH, 1), (BlockOp::SplitH, 1),
    (BlockOp::SplitH, 1), (BlockOp::SplitH, 1)
  ], [
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::Skip,   3), (BlockOp::SplitH, 3)
  ]
];
static H6CB_4X2: [H6CB; 2] = [
  [
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::Motion, 1), (BlockOp::Motion, 1),
    (BlockOp::SplitV, 1), (BlockOp::SplitV, 1),
    (BlockOp::SplitV, 1), (BlockOp::SplitV, 1)
  ], [
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::Skip,   3), (BlockOp::SplitV, 3)
  ]
];
static H6CB_2X2: H6CB = [
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::ShortM, 1), (BlockOp::ShortM, 1),
    (BlockOp::Motion, 2), (BlockOp::Motion, 2),
    (BlockOp::Skip,   2), (BlockOp::Skip,   2)
];

const LUMA_QMAT: [u8; 64] = [
    16,  11,  10,  16,  24,  40,  51,  61,
    12,  12,  14,  19,  26,  58,  60,  55,
    14,  13,  16,  24,  40,  57,  69,  56,
    14,  17,  22,  29,  51,  87,  80,  62,
    18,  22,  37,  56,  68, 109, 103,  77,
    24,  35,  55,  64,  81, 104, 113,  92,
    49,  64,  78,  87, 103, 121, 120, 101,
    72,  92,  95,  98, 112, 100, 103,  99
];
const CHROMA_QMAT: [u8; 64] = [
    17,  18,  24,  47,  99,  99,  99,  99,
    18,  21,  26,  66,  99,  99,  99,  99,
    24,  26,  56,  99,  99,  99,  99,  99,
    47,  66,  99,  99,  99,  99,  99,  99,
    99,  99,  99,  99,  99,  99,  99,  99,
    99,  99,  99,  99,  99,  99,  99,  99,
    99,  99,  99,  99,  99,  99,  99,  99,
    99,  99,  99,  99,  99,  99,  99,  99
];
