use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

struct GuiltyDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      [u8; WIDTH * HEIGHT],
    vdata:      Vec<u8>,
    nfrm:       u16,
    end:        u64,
}

impl GuiltyDecoder {
    fn unpack_i(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let mut dpos = 0;
        while dpos < self.frame.len() {
            let op = br.read_byte()?;
            if (op & 0xC0) == 0xC0 {
                let len = (op & 0x3F) as usize;
                validate!(dpos + len <= self.frame.len());
                let clr = br.read_byte()?;
                for el in self.frame[dpos..][..len].iter_mut() {
                    *el = clr;
                }
                dpos += len;
            } else {
                self.frame[dpos] = op;
                dpos += 1;
            }
        }
        validate!(br.left() == 768 + 1);
        validate!(br.read_byte()? == 0x0C);
        br.read_buf(&mut self.pal)?;

        Ok(())
    }
    fn unpack_p(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let short_skip_tag = br.read_byte()?;
        let long_skip_tag  = br.read_byte()?;

        let mut dpos = 0;
        while dpos < self.frame.len() {
            let op = br.read_byte()?;
            if op == short_skip_tag {
                let len = br.read_byte()? as usize;
                validate!(dpos + len <= self.frame.len());
                if len > 0 {
                    dpos += len;
                } else {
                    self.frame[dpos] = op;
                    dpos += 1;
                }
            } else if op == long_skip_tag {
                let len = br.read_u16le()? as usize;
                validate!(dpos + len <= self.frame.len());
                if len > 0 {
                    dpos += len;
                } else {
                    self.frame[dpos] = op;
                    dpos += 1;
                }
            } else {
                self.frame[dpos] = op;
                dpos += 1;
            }
        }
        validate!(br.left() == 0);

        Ok(())
    }
}

impl InputSource for GuiltyDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 12,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        if self.nfrm == 0 {
            validate!(br.tell() == self.end);
            let chunk_size = br.read_u32le().map_err(|_| DecoderError::EOF)?;
            validate!(chunk_size > 0);
            self.nfrm = br.read_u16le()?;
            validate!(self.nfrm > 0);
            self.end = br.tell() + u64::from(chunk_size);

            let frm_size = br.read_u16le()? as usize;
            validate!(frm_size > 4);
            let width = br.read_u16le()? as usize;
            let height = br.read_u16le()? as usize;
            validate!(width == WIDTH && height == HEIGHT);
            self.vdata.resize(frm_size - 4, 0);
            br.read_buf(&mut self.vdata)?;
            self.unpack_i().map_err(|_| DecoderError::InvalidData)?;
        } else {
            let frm_size = br.read_u16le()? as usize;
            validate!(frm_size > 2);
            self.vdata.resize(frm_size, 0);
            br.read_buf(&mut self.vdata)?;
            self.unpack_p().map_err(|_| DecoderError::InvalidData)?;
        }

        let pal = self.pal;
        let frame = self.frame.to_vec();
        self.nfrm -= 1;
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(file);
    Ok(Box::new(GuiltyDecoder {
        fr,
        pal:    [0; 768],
        vdata:  Vec::with_capacity(WIDTH * HEIGHT),
        frame:  [0; WIDTH * HEIGHT],
        nfrm:   0,
        end:    0,
    }))
}
