use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const DICT_SIZE: usize = 4096;
const MAX_BITS:   u8 = 12;
const INVALID_POS: usize = 65536;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
    nsyms:      usize,
    idx_bits:   u8,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
            idx_bits:   0,
            nsyms:      0,
        }
    }
    fn reset(&mut self, bits: u8) {
        self.nsyms    = 1 << bits;
        self.dict_pos = self.nsyms + 2;
        self.dict_lim = 1 << (bits + 1);
        self.idx_bits = bits + 1;
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut Vec<u8>, pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        for _ in 0..tot_len {
            dst.push(0);
        }

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() > 1);
        let mut br = BitReader::new(src, BitReaderMode::LE);

        dst.clear();

        self.reset(8);
        let firstidx = br.read(self.idx_bits).map_err(|_| DecoderError::InvalidData)? as usize;
        validate!(firstidx == 0x100);
        'restart: loop {
            let mut lastidx = br.read(self.idx_bits).map_err(|_| DecoderError::InvalidData)? as usize;
            if lastidx == 0x101 {
                return Ok(());
            }
            validate!(lastidx < 0x100);
            dst.push(lastidx as u8);
            loop {
                let ret         = br.read(self.idx_bits);
                if ret.is_err() {
                    return Ok(());
                }
                let idx = ret.unwrap() as usize;
                match idx {
                    0x100 => {
                        self.reset(8);
                        continue 'restart;
                    },
                    0x101 => return Ok(()),
                    _ => {},
                }
                validate!(idx <= self.dict_pos);
                let pos = dst.len();
                if idx != self.dict_pos {
                    self.decode_idx(dst, pos, idx)?;
                    self.add(lastidx, dst[pos]);
                } else {
                    self.decode_idx(dst, pos, lastidx)?;
                    let lastsym = dst[pos];
                    dst.push(lastsym);
                    self.add(lastidx, lastsym);
                }
                lastidx = idx;

                if self.dict_pos == DICT_SIZE {
                    self.reset(8);
                    let idx = br.read(self.idx_bits)?;
                    match idx {
                        0x100 => {
                            continue 'restart;
                        },
                        0x101 => return Ok(()),
                        _ => return Err(DecoderError::InvalidData),
                    }
                }
                if self.dict_pos == self.dict_lim {
                    self.dict_lim <<= 1;
                    self.idx_bits += 1;
                }
            }
        }
    }
}

struct FrameRecord {
    ops_offset: u32,
    ops_size:   usize,
    pix_offset: u32,
    pix_size:   usize,
    snd_offset: u32,
    snd_size:   usize,
}

impl FrameRecord {
    fn read(br: &mut dyn ByteIO) -> DecoderResult<Self> {
        br.read_skip(8)?;
        let ops_offset = br.read_u32le()?;
        let ops_size = br.read_u32le()? as usize;
        br.read_skip(8)?;
        let pix_offset = br.read_u32le()?;
        let pix_size = br.read_u32le()? as usize;
        let snd_offset = br.read_u32le()?;
        let snd_size = br.read_u32le()? as usize;
        let next_pos = br.read_u32le()?;
        validate!(ops_offset <= next_pos && (next_pos - ops_offset) as usize >= ops_size);
        validate!(pix_offset <= next_pos && (next_pos - pix_offset) as usize >= pix_size);
        let _left = usize::from(br.read_u16le()?);
        let _top = usize::from(br.read_u16le()?);
        let _right = usize::from(br.read_u16le()?);
        let _bottom = usize::from(br.read_u16le()?);
        //validate!(left <= right && top <= bottom);
        br.read_skip(12)?;
        Ok(Self { ops_offset, ops_size, pix_offset, pix_size, snd_offset, snd_size })
    }
}

struct FLKDecoder {
    fr:         FileReader<BufReader<File>>,
    base:       u64,
    end:        u64,
    width:      usize,
    height:     usize,
    frame:      Vec<u16>,
    frm_rec:    Vec<FrameRecord>,
    cur_frm:    usize,
    data:       Vec<u8>,
    pix_data:   Vec<u8>,
    lzw:        LZWState,
    pal:        [u16; 256],
    audio:      bool,
    has_audio:  bool,
}

#[derive(Clone,Copy,Debug,PartialEq)]
enum OpMode {
    None,
    Run(u16, u16),
    Raw(u16),
    Bkg(u16),
    Skip(u16),
    LineSkip(u8),
}

fn unpack_frame(ops_data: &[u8], pix_data: &[u8], pal: &[u16; 256], frame: &mut [u16], stride: usize) -> DecoderResult<()> {
    let mut ops = MemoryReader::new_read(ops_data);
    let mut pix = MemoryReader::new_read(pix_data);

    let mut op_mode = OpMode::None;
    for strip in frame.chunks_exact_mut(stride) {
        if let OpMode::LineSkip(nlines) = op_mode {
            op_mode = if nlines > 1 { OpMode::LineSkip(nlines - 1) } else { OpMode::None };
            if nlines > 0 {
                continue;
            }
        }

        let mut trunc = false;
        for (x, dst) in strip.iter_mut().enumerate() {
            if op_mode == OpMode::None {
                let op = ops.read_byte()?;
                if op == 0xC0 {
                    let line_op = ops.read_byte()?;
                    if line_op == 0xFF {
                        return Ok(());
                    }
                    if line_op > 0 {
                        op_mode = OpMode::LineSkip(if x > 0 { line_op } else { line_op - 1});
                    }
                    trunc = true;
                    break;
                }
                let mut len = u16::from(op & 0x1F);
                if (op & 0x20) != 0 {
                    len = (len << 8) | u16::from(ops.read_byte()?);
                }
                validate!(len > 0);
                op_mode = match op >> 6 {
                        0 => OpMode::Raw(len),
                        1 => OpMode::Run(len, pal[usize::from(pix.read_byte()?)]),
                        2 => OpMode::Bkg(len),
                        _ => OpMode::Skip(len),
                    };
            }

            op_mode = match op_mode {
                OpMode::Raw(len) => {
                    *dst = pal[usize::from(pix.read_byte()?)];
                    if len > 1 { OpMode::Raw(len - 1) } else { OpMode::None }
                },
                OpMode::Run(len, pix) => {
                    *dst = pix;
                    if len > 1 { OpMode::Run(len - 1, pix) } else { OpMode::None }
                },
                OpMode::Bkg(len) => {
                    // since we don't have an external background, we assume it to be black
                    *dst = 0;
                    if len > 1 { OpMode::Bkg(len - 1) } else { OpMode::None }
                },
                OpMode::Skip(len) => {
                    if len > 1 { OpMode::Skip(len - 1) } else { OpMode::None }
                },
                _ => unreachable!(),
            };
        }

        if !trunc {
            let op = ops.read_byte()?;
            validate!(op == 0xC0);
            let line_op = ops.read_byte()?;
            if line_op == 0xFF {
                return Ok(());
            }
            if line_op > 0 {
                op_mode = OpMode::LineSkip(line_op);
            }
        }
    }

    let op = ops.read_byte()?;
    validate!(op == 0xC0);
    let line_op = ops.read_byte()?;
    validate!(line_op == 0xFF);

    Ok(())
}

impl InputSource for FLKDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 12,
                }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    sample_type: AudioSample::S16,
                    channels:    1,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        if self.cur_frm >= self.frm_rec.len() {
            validate!(br.tell() <= self.end);
            br.seek(SeekFrom::Start(self.end))?;

            let nframes = br.read_u32le().map_err(|_| DecoderError::EOF)? as usize;
            validate!((1..=2000).contains(&nframes));
            let video_size = br.read_u32le()?;
            let width = br.read_u32le()? as usize;
            let height = br.read_u32le()? as usize;
            validate!(self.width == width && self.height == height);
            let toc_size = br.read_u32le()? as usize;
            validate!(toc_size == nframes * 0x40);
            br.read_u32le()?;
            br.read_u32le()?;
            br.read_u32le()?;
            let unp_size = br.read_u32le()? as usize;
            validate!(unp_size < 1048576);
            br.read_u32le()?;
            br.read_u32le()?;
            br.read_u32le()?;
            br.read_u32le()?;
            self.base = br.tell();

            self.frm_rec.clear();
            for _ in 0..nframes {
                let rec = FrameRecord::read(br)?;
                self.frm_rec.push(rec);
            }
            self.cur_frm = 0;
            self.end = self.base + u64::from(video_size);

            for el in self.pal.iter_mut() {
                *el = br.read_u16le().unwrap_or_default();
            }
            self.audio = self.has_audio;
        }
        validate!(self.cur_frm < self.frm_rec.len());

        let frm_info = &self.frm_rec[self.cur_frm];

        if self.audio {
            self.audio = false;
            if frm_info.snd_size > 0 {
                validate!(frm_info.snd_size == 0x72E);
                br.seek(SeekFrom::Start(self.base + u64::from(frm_info.snd_offset)))?;
                let mut snd = vec![0; frm_info.snd_size / 2];
                for el in snd.iter_mut() {
                    *el = br.read_u16le()? as i16;
                }
                return Ok((1, Frame::AudioS16(snd)));
            }
        }

        br.seek(SeekFrom::Start(self.base + u64::from(frm_info.pix_offset)))?;
        self.data.resize(frm_info.pix_size, 0);
        if !self.data.is_empty() {
            br.read_buf(&mut self.data)?;
            self.lzw.unpack(&self.data, &mut self.pix_data)
                .map_err(|_| DecoderError::InvalidData)?;
        }
        if frm_info.ops_size > 0 {
            br.seek(SeekFrom::Start(self.base + u64::from(frm_info.ops_offset)))?;
            self.data.resize(frm_info.ops_size, 0);
            br.read_buf(&mut self.data)?;

            unpack_frame(&self.data, &self.pix_data, &self.pal, &mut self.frame, self.width)
                .map_err(|_| DecoderError::InvalidData)?;
        } else {
            validate!(self.pix_data.len() >= self.width * self.height);
            for (dst, &pix) in self.frame.iter_mut().zip(self.pix_data.iter()) {
                *dst = self.pal[usize::from(pix)];
            }
        }

        self.cur_frm += 1;
        self.audio = self.has_audio;

        Ok((0, Frame::VideoRGB16(self.frame.clone())))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let nframes = fr.read_u32le()? as usize;
    validate!((1..=2000).contains(&nframes));
    let video_size = fr.read_u32le()?;
    let width = fr.read_u32le()? as usize;
    let height = fr.read_u32le()? as usize;
    validate!((1..=1024).contains(&width) && (1..=1024).contains(&height));
    let toc_size = fr.read_u32le()? as usize;
    validate!(toc_size == nframes * 0x40);
    fr.read_u32le()?;
    fr.read_u32le()?;
    fr.read_u32le()?;
    let unp_size = fr.read_u32le()? as usize;
    validate!(unp_size < 1048576);
    fr.read_u32le()?;
    fr.read_u32le()?;
    fr.read_u32le()?;
    fr.read_u32le()?;

    let mut frm_rec = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let rec = FrameRecord::read(&mut fr)?;
        frm_rec.push(rec);
    }
    validate!(!frm_rec.is_empty());

    let has_audio = frm_rec[0].snd_size > 0;

    let mut pal = [0; 256];
    // some files may be shorter than expected, so ignore palette reading errors
    for el in pal.iter_mut() {
        *el = fr.read_u16le().unwrap_or_default();
    }

    Ok(Box::new(FLKDecoder {
        fr,
        width, height, pal,
        frame: vec![0; width * height],
        frm_rec,
        cur_frm: 0,
        data: Vec::new(),
        pix_data: Vec::with_capacity(unp_size),
        lzw: LZWState::new(),
        base: 0x34,
        end: u64::from(video_size) + 0x34,
        has_audio,
        audio: has_audio,
    }))
}
