use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::rnc::rnc_uncompress;

struct VecReader {
    buf:    Vec<u8>,
    pos:    usize,
}

impl VecReader {
    fn new_read(buf: Vec<u8>) -> Self {
        Self { buf, pos: 0 }
    }

    fn real_seek(&mut self, pos: i64) -> ByteIOResult<u64> {
        if pos < 0 || (pos as usize) > self.buf.len() {
            return Err(ByteIOError::WrongRange);
        }
        self.pos = pos as usize;
        Ok(pos as u64)
    }
}

impl ByteIO for VecReader {
    fn read_byte(&mut self) -> ByteIOResult<u8> {
        if self.is_eof() { return Err(ByteIOError::EOF); }
        let res = self.buf[self.pos];
        self.pos += 1;
        Ok(res)
    }

    fn peek_byte(&mut self) -> ByteIOResult<u8> {
        if self.is_eof() { return Err(ByteIOError::EOF); }
        Ok(self.buf[self.pos])
    }

    fn peek_buf(&mut self, buf: &mut [u8]) -> ByteIOResult<usize> {
        let copy_size = if self.buf.len() - self.pos < buf.len() { self.buf.len() - self.pos } else { buf.len() };
        if copy_size == 0 { return Err(ByteIOError::EOF); }
        let dst = &mut buf[0..copy_size];
        dst.copy_from_slice(&self.buf[self.pos..][..copy_size]);
        Ok(copy_size)
    }

    fn read_buf(&mut self, buf: &mut [u8]) -> ByteIOResult<()> {
        let read_size = self.peek_buf(buf)?;
        if read_size < buf.len() { return Err(ByteIOError::EOF); }
        self.pos += read_size;
        Ok(())
    }

    fn read_buf_some(&mut self, buf: &mut [u8]) -> ByteIOResult<usize> {
        let read_size = self.peek_buf(buf)?;
        self.pos += read_size;
        Ok(read_size)
    }

    fn write_buf(&mut self, _buf: &[u8]) -> ByteIOResult<()> {
        Err(ByteIOError::NotImplemented)
    }

    fn tell(&mut self) -> u64 {
        self.pos as u64
    }

    fn seek(&mut self, pos: SeekFrom) -> ByteIOResult<u64> {
        let cur_pos  = self.pos       as i64;
        let cur_size = self.buf.len() as i64;
        match pos {
            SeekFrom::Start(x)   => self.real_seek(x as i64),
            SeekFrom::Current(x) => self.real_seek(cur_pos + x),
            SeekFrom::End(x)     => self.real_seek(cur_size + x),
        }
    }

    fn is_eof(&self) -> bool {
        self.pos >= self.buf.len()
    }

    fn is_seekable(&mut self) -> bool {
        true
    }

    fn size(&mut self) -> i64 {
        self.buf.len() as i64
    }

    fn flush(&mut self) -> ByteIOResult<()> { Ok(()) }
}

#[derive(Debug,PartialEq)]
enum FlicType {
    AlienVirus,
    Bureau13,
    CFO,
    FLI,
    FLC,
    FLH,
    MagicCarpet,
    Origin,
    PTF
}

struct VideoData {
    frame:      Vec<u8>,
    frame16:    Vec<u16>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    cur_w:      usize,
    cur_h:      usize,
    depth:      u16,
    unp_buf:    Vec<u8>,
    vdata:      Vec<u8>,
}

macro_rules! decode_blocks {
    ($self:expr, $row:expr, $br:expr, $blkfunc:ident, $name:expr) => {
        let nsegs = $br.read_u16le()? as i16;
        let mut skip_first = nsegs > 0;
        let nsegs = (nsegs.unsigned_abs() as usize + 1) / 2;
        let mut offset = 0;
        for _ in 0..nsegs {
            offset += if !skip_first {
                    usize::from($br.read_byte()?)
                } else {
                    skip_first = false;
                    0
                };
            let nblks = usize::from($br.read_byte()?);
            validate!(offset + nblks <= ($self.cur_w / 4));
            for pos_x in offset..(nblks + offset) {
                Self::$blkfunc($br, &mut $row[pos_x * 4..], $self.width)?;
            }
            offset += nblks;
        }
    }
}

impl VideoData {
    fn handle_subchunk(&mut self, br: &mut dyn ByteIO, stype: u16, ssize: usize) -> DecoderResult<()> {
        let _schunk_end = br.tell() + (ssize as u64);
        match stype {
            3 => return Err(DecoderError::NotImplemented), // CEL data
            4 => { // 256-level palette
                let num_upds = br.read_u16le()? as usize;
                let mut idx = 0;
                for _ in 0..num_upds {
                    idx += usize::from(br.read_byte()?);
                    let mut len = usize::from(br.read_byte()?);
                    if len == 0 {
                        len = 256;
                    }
                    validate!(idx + len <= 256);
                    br.read_buf(&mut self.pal[idx * 3..][..len * 3])?;
                    idx += len;
                }
            },
            7 => { // delta FLC
                validate!(self.depth <= 8);
                let mut tot_lines = br.read_u16le()? as usize;
                validate!(tot_lines <= self.cur_h);
                let mut skip = 0;
                for (y, line) in self.frame.chunks_exact_mut(self.width).take(self.cur_h).enumerate() {
                    if skip > 0 {
                        skip -= 1;
                        continue;
                    }
                    let opcode = br.read_u16le()?;
                    match opcode >> 14 {
                        0 => {
                            let nops = opcode as usize;
                            let mut pos = 0;
                            for _ in 0..nops {
                                let skip = usize::from(br.read_byte()?);
                                let op = usize::from(br.read_byte()?);
                                let len = (if (op & 0x80) == 0 { op } else { 256 - op }) * 2;
                                validate!(pos + skip + len <= self.cur_w);
                                pos += skip;
                                if (op & 0x80) == 0 {
                                    br.read_buf(&mut line[pos..][..len])?;
                                } else {
                                    let mut c = [0; 2];
                                    br.read_buf(&mut c)?;
                                    for pair in line[pos..][..len].chunks_exact_mut(2) {
                                        pair.copy_from_slice(&c);
                                    }
                                }
                                pos += len;
                            }
                            tot_lines -= 1;
                            if tot_lines == 0 {
                                break;
                            }
                        },
                        2 => {
                            line[self.cur_w - 1] = opcode as u8;
                            tot_lines -= 1;
                            if tot_lines == 0 {
                                break;
                            }
                        },
                        3 => {
                            skip = 0x10000 - (opcode as usize);
                            validate!(skip > 0 && y + skip <= self.cur_h);
                            skip -= 1;
                        },
                        _ => return Err(DecoderError::InvalidData),
                    }
                }
            },
            11 => { // VGA palette
                let num_upds = br.read_u16le()? as usize;
                let mut idx = 0;
                for _ in 0..num_upds {
                    idx += usize::from(br.read_byte()?);
                    let mut len = usize::from(br.read_byte()?);
                    if len == 0 {
                        len = 256;
                    }
                    validate!(idx + len <= 256);
                    br.read_vga_pal_some(&mut self.pal[idx * 3..][..len * 3])?;
                    idx += len;
                }
            },
            12 => { // FLI_LC
                validate!(self.depth <= 8);
                let start = br.read_u16le()? as usize;
                let nlines = br.read_u16le()? as usize;
                validate!(start + nlines <= self.cur_h);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h).skip(start).take(nlines) {
                    let npkts = usize::from(br.read_byte()?);
                    let mut pos = 0;
                    for _ in 0..npkts {
                        let skip = usize::from(br.read_byte()?);
                        let op = usize::from(br.read_byte()?);
                        let len = if (op & 0x80) == 0 { op } else { 256 - op };
                        validate!(pos + skip + len <= self.cur_w);
                        pos += skip;
                        if (op & 0x80) == 0 {
                            br.read_buf(&mut line[pos..][..len])?;
                        } else {
                            let c = br.read_byte()?;
                            for el in line[pos..][..len].iter_mut() {
                                *el = c;
                            }
                        }
                        pos += len;
                    }
                }
            },
            13 => { // black frame
                validate!(ssize == 0);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    for el in line[..self.cur_w].iter_mut() {
                        *el = 0;
                    }
                }
            },
            15 => { // FLI_BRUN
                validate!(self.depth <= 8);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    let _npkts = br.read_byte()?;
                    let mut pos = 0;
                    while pos < self.cur_w {
                        let op = usize::from(br.read_byte()?);
                        if (op & 0x80) == 0 {
                            validate!(pos + op <= self.cur_w);
                            let c = br.read_byte()?;
                            for el in line[pos..][..op].iter_mut() {
                                *el = c;
                            }
                            pos += op;
                        } else {
                            let len = 256 - op;
                            validate!(pos + len <= self.cur_w);
                            br.read_buf(&mut line[pos..][..len])?;
                            pos += len;
                        }
                    }
                }
            },
            16 => { // raw image
                validate!(self.depth <= 8);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    br.read_buf(&mut line[..self.cur_w])?;
                }
            },
            18 => br.read_skip(ssize)?, // postage stamp aka thumbnail
            25 => { // DTA_BRUN
                validate!(self.depth == 15 || self.depth == 16);
                for line in self.frame16.chunks_exact_mut(self.width).take(self.cur_h) {
                    let _npkts = br.read_byte()?;
                    let mut pos = 0;
                    while pos < self.cur_w {
                        let op = usize::from(br.read_byte()?);
                        if (op & 0x80) == 0 {
                            validate!(pos + op <= self.cur_w);
                            let c = br.read_u16le()?;
                            for el in line[pos..][..op].iter_mut() {
                                *el = c;
                            }
                            pos += op;
                        } else {
                            let len = 256 - op;
                            validate!(pos + len <= self.cur_w);
                            for el in line[pos..][..len].iter_mut() {
                                *el = br.read_u16le()?;
                            }
                            pos += len;
                        }
                    }
                }
            },
            26 => { // DTA_COPY
                validate!(self.depth == 15 || self.depth == 16);
                for line in self.frame16.chunks_exact_mut(self.width).take(self.cur_h) {
                    for el in line[..self.cur_w].iter_mut() {
                        *el = br.read_u16le()?;
                    }
                }
            },
            27 => { // DTA_LC
                validate!(self.depth == 15 || self.depth == 16);
                let mut tot_lines = br.read_u16le()? as usize;
                validate!(tot_lines <= self.cur_h);
                let mut skip = 0;
                for (y, line) in self.frame16.chunks_exact_mut(self.width).take(self.cur_h).enumerate() {
                    if skip > 0 {
                        skip -= 1;
                        continue;
                    }
                    let opcode = br.read_u16le()? as i16;
                    if opcode > 0 {
                        let nops = opcode as usize;
                        let mut pos = 0;
                        for _ in 0..nops {
                            let skip = usize::from(br.read_byte()?);
                            let op = usize::from(br.read_byte()?);
                            let len = if (op & 0x80) == 0 { op } else { 256 - op };
                            validate!(pos + skip + len <= self.cur_w);
                            pos += skip;
                            if (op & 0x80) == 0 {
                                for el in line[pos..][..len].iter_mut() {
                                    *el = br.read_u16le()?;
                                }
                            } else {
                                let c = br.read_u16le()?;
                                for el in line[pos..][..len].iter_mut() {
                                    *el = c;
                                }
                            }
                            pos += len;
                        }
                        tot_lines -= 1;
                        if tot_lines == 0 {
                            break;
                        }
                    } else {
                        skip = (-opcode) as usize;
                        validate!(skip > 0 && y + skip <= self.cur_h);
                        skip -= 1;
                    }
                }
            },
            31 => br.read_skip(ssize)?, // frame label
            32 => br.read_skip(ssize)?, // bitmap mask
            33 => br.read_skip(ssize)?, // multilevel mask
            34 => br.read_skip(ssize)?, // segment information
            35 => br.read_skip(ssize)?, // key image
            36 => br.read_skip(ssize)?, // key palette
            37 => br.read_skip(ssize)?, // region of frame differences
            38 => return Err(DecoderError::NotImplemented), // wave
            39 => br.read_skip(ssize)?, // user string
            40 => br.read_skip(ssize)?, // region mask
            41 => br.read_skip(ssize)?, // extended frame label
            42 => br.read_skip(ssize)?, // scanline delta shifts
            43 => br.read_skip(ssize)?, // path map
            0x5555 => {
                self.vdata.resize(ssize, 0);
                br.read_buf(&mut self.vdata)?;
                rnc_uncompress(&self.vdata, &mut self.unp_buf)?;
                for (dst, src) in self.frame16.iter_mut().zip(self.unp_buf.chunks_exact(2)) {
                    *dst = read_u16le(src).unwrap_or_default();
                }
            },
            _ => br.read_skip(ssize)?,
        }
        Ok(())
    }

    fn fill_block(br: &mut dyn ByteIO, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let c = br.read_byte()?;
        for line in dst.chunks_mut(stride) {
            for el in line[..4].iter_mut() {
                *el = c;
            }
        }
        Ok(())
    }
    fn block_2clr(br: &mut dyn ByteIO, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let mut clr = [0; 2];
        br.read_buf(&mut clr)?;
        let mut mask = br.read_u16le()? as usize;
        for line in dst.chunks_mut(stride) {
            for el in line[..4].iter_mut() {
                *el = clr[mask & 1];
                mask >>= 1;
            }
        }
        Ok(())
    }
    fn raw_block(br: &mut dyn ByteIO, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        for line in dst.chunks_mut(stride) {
            br.read_buf(&mut line[..4])?;
        }
        Ok(())
    }
    fn block_8clr(br: &mut dyn ByteIO, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let mut clr = [0; 2];
        let mut mask = br.read_u16le()? as usize;
        for line in dst.chunks_mut(stride) {
            br.read_buf(&mut clr)?;
            for el in line[..4].iter_mut() {
                *el = clr[mask & 1];
                mask >>= 1;
            }
        }
        Ok(())
    }
    fn do_0b1c_frame(&mut self, br: &mut dyn ByteIO) -> DecoderResult<()> {
        validate!((self.cur_w & 3) == 0 && (self.cur_h & 3) == 0);
        for row in self.frame.chunks_exact_mut(self.width * 4).take(self.cur_h / 4) {
            let flags = br.read_byte()?;
            if (flags & 1) != 0 {
                decode_blocks!(self, row, br, fill_block, "fill");
            }
            if (flags & 2) != 0 {
                decode_blocks!(self, row, br, block_2clr, "2clr");
            }
            if (flags & 4) != 0 {
                decode_blocks!(self, row, br, raw_block, "raw");
            }
            if (flags & 8) != 0 {
                decode_blocks!(self, row, br, block_8clr, "8clr");
            }
        }

        Ok(())
    }
}

struct B13Data {
    vbuf:       Vec<u8>,
    orig_w:     usize,
    orig_h:     usize,
    abuf:       Vec<u8>,
    arate:      u32,
    abuf_len:   usize,
    next_sync:  u64,
}

enum CustomData {
    None,
    B13(B13Data),
}

struct FlicDecoder {
    fr:         Box<dyn ByteIO>,
    frm_no:     u16,
    nframes:    u16,
    fli_end:    u64,
    tot_size:   u64,
    speed:      u32,
    video:      VideoData,
    audio16:    Vec<i16>,
    firstaud:   bool,
    aligned:    bool,
    subtype:    FlicType,
    custom:     CustomData,
}

impl InputSource for FlicDecoder {
    fn get_num_streams(&self) -> usize {
        let mut nstreams = 0;
        if self.video.width > 0 && self.video.height > 0 {
            nstreams += 1;
        }
        if self.subtype == FlicType::PTF {
            nstreams += 1;
        }
        if self.subtype == FlicType::FLH {
            nstreams += 1;
        }
        if let CustomData::B13(ref data) = self.custom {
            if data.arate > 0 && data.abuf_len > 0 {
                nstreams += 1;
            }
        }
        nstreams
    }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        let last_id = self.get_num_streams() - 1;
        if let CustomData::B13(ref data) = self.custom {
            if stream_no == last_id && data.arate > 0 && data.abuf_len > 0 {
                return StreamInfo::Audio(AudioInfo{
                    sample_rate: data.arate,
                    sample_type: AudioSample::U8,
                    channels:    1,
                });
            }
        }
        if (self.subtype == FlicType::PTF || self.subtype == FlicType::FLH) && stream_no == last_id {
            StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::S16,
                channels:    1,
            })
        } else if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.video.width,
                height: self.video.height,
                bpp:    self.video.depth as u8,
                tb_num: self.speed,
                tb_den: 1000,
            })
        } else {
            StreamInfo::None
        }
    }
    #[allow(clippy::collapsible_if)]
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.audio16.is_empty() {
            let mut ret = Vec::new();
            std::mem::swap(&mut ret, &mut self.audio16);
            return Ok((1, Frame::AudioS16(ret)));
        }
        if let CustomData::B13(ref mut data) = self.custom {
            if !data.abuf.is_empty() {
                let mut ret = Vec::new();
                std::mem::swap(&mut ret, &mut data.abuf);
                return Ok((1, Frame::AudioU8(ret)));
            }
        }
        let br = &mut self.fr;
        if self.subtype == FlicType::Origin {
            if br.tell() >= self.tot_size {
                return Err(DecoderError::EOF);
            }
            if br.tell() >= self.fli_end {
                if (br.tell() & 1) != 0 {
                    br.read_skip(1)?;
                }
                let tag = br.read_tag()?;
                // we can't handle FONT and VOCF chunks (yet?)
                if &tag != b"FLIC" {
                    return Err(DecoderError::EOF);
                }
                let size = br.read_u32be()?;
                validate!(size > 0x8C);
                self.fli_end = br.tell() + u64::from(size);
                br.read_skip(12)?;

                let tag         = br.read_u16le()?;
                validate!(tag == 0xAF11);
                let _nframes    = br.read_u16le()?;
                let width       = br.read_u16le()? as usize;
                let height      = br.read_u16le()? as usize;
                validate!(width == self.video.width && height == self.video.height);
                let depth       = br.read_u16le()?;
                validate!(depth == 8);
                let _flags      = br.read_u16le()?;
                let _speed      = br.read_u32le()?;
                                  br.read_skip(0x6C)?;
                self.frm_no = 0;
                self.nframes = 32000;
            }
        }
        if self.frm_no >= self.nframes && self.subtype != FlicType::Bureau13 {
            return Err(DecoderError::EOF);
        }
        if let CustomData::B13(ref mut data) = self.custom {
            if br.tell() >= self.fli_end {
                loop {
                    if br.tell() == data.next_sync {
                        let _next_asize = br.read_u32le().map_err(|_| DecoderError::EOF)?;
                        let next_csize = br.read_u32le()?;
                        data.next_sync = br.tell() + u64::from(next_csize);
                    }
                    let ctype = br.read_byte().map_err(|_| DecoderError::EOF)?;
                                br.read_byte()?;
                    let csize = br.read_u32le()?;
                    match ctype {
                        0x02 => { // todo audio
                            if csize > 0 && data.abuf_len > 0 {
                                br.read_extend(&mut data.abuf, csize as usize)?;
                            } else {
                                br.read_skip(csize as usize)?;
                            }
                        },
                        0x03 => {
                            validate!(csize >= 0x80);
                            let _size    = br.read_u32le()?;
                            let tag      = br.read_u16le()?;
                            validate!(tag == 0xAF11 || tag == 0xAF12);
                            let _nframes = br.read_u16le()?;
                            let width    = br.read_u16le()? as usize;
                            let height   = br.read_u16le()? as usize;
                            if width > data.orig_w || height > data.orig_h {
                                println!("size change to {width}x{height}");
                                return Err(DecoderError::NotImplemented);
                            }
                            self.video.width  = width;
                            self.video.height = height;
                            for el in data.vbuf.iter_mut() {
                                *el = 0;
                            }
                            br.read_skip(csize as usize - 12)?;
                        },
                        0x04 => {
                            self.fli_end = br.tell() + u64::from(csize);
                            break;
                        },
                        _ => { br.read_skip(csize as usize)?; },
                    }
                }
            }
        }
        while br.tell() < self.fli_end {
            let chunk_size = br.read_u32le()?;
            let chunk_type = br.read_u16le()?;
            //println!("chunk {chunk_type:X} size {chunk_size:X} @ {:X}", br.tell() - 6);
            let (chunk_size, pal_flag) = if self.subtype == FlicType::CFO {
                    (chunk_size as usize + 2, false)
                } else if self.subtype != FlicType::PTF {
                    validate!(chunk_size >= 6);
                    (chunk_size as usize - 6, false)
                } else if (chunk_size & 0x80000000) == 0 {
                    if chunk_type != 0x0B1C && chunk_size >= 6 {
                        (chunk_size as usize - 6, false)
                    } else {
                        (chunk_size as usize, false)
                    }
                } else {
                    validate!(chunk_type == 0x0B1C);
                    ((-(chunk_size as i32)) as usize + 0x300, true)
                };
            let chunk_end = br.tell() + (chunk_size as u64);
            if self.aligned {
                if chunk_size & 1 != 0 {
                    println!("FLIC seems to contain unaligned chunks");
                    self.aligned = false;
                }
            }
            match self.subtype {
                FlicType::FLI | FlicType::FLC | FlicType::MagicCarpet | FlicType::Origin | FlicType::CFO | FlicType::AlienVirus | FlicType::Bureau13 => {
                    validate!(chunk_type >= 0xF100);
                },
                FlicType::PTF => {},
                FlicType::FLH => {
                    validate!(chunk_type >= 0xF100 || chunk_type == 0x4B4C);
                },
            }
            validate!(chunk_end <= self.fli_end);
            match chunk_type {
                0x0B1C => {
                    validate!(self.subtype == FlicType::PTF);
                    if pal_flag {
                        br.read_vga_pal(&mut self.video.pal)?;
                    }
                    if chunk_size > 0 {
                        self.video.do_0b1c_frame(br.as_mut())?;
                    }
                    validate!(br.tell() == chunk_end);
                    self.frm_no += 1;

                    let pal = self.video.pal;
                    let frame = self.video.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x5657 => { // WAV
                    validate!(self.subtype == FlicType::PTF);
                    if chunk_size < 6 {
                        self.audio16.resize(chunk_size / 2, 0);
                    } else if self.firstaud {
                        self.firstaud = false;
                        br.read_skip(44)?; // RIFF header
                        self.audio16.resize((chunk_size + 6 - 44) / 2, 0);
                    } else {
                        self.audio16.resize((chunk_size + 6) / 2, 0);
                    }
                    for el in self.audio16.iter_mut() {
                        *el = br.read_u16le()? as i16;
                    }
                },
                0xF100 => br.read_skip(chunk_size)?, // CEL data holder or some unknown chunk
                0xF1E0 => br.read_skip(chunk_size)?, // script chunk
                0xF1FA => {
                    let chunks = br.read_u16le()? as usize;
                    validate!(chunks * 6 <= chunk_size);
                    if self.subtype != FlicType::CFO {
                        let _delay = br.read_u16le()?;
                        let _zero  = br.read_u16le()?;
                        let cur_w  = br.read_u16le()? as usize;
                        let cur_h  = br.read_u16le()? as usize;
                        validate!(cur_w <= self.video.width && cur_h <= self.video.height);
                        self.video.cur_w = if cur_w > 0 { cur_w } else { self.video.width };
                        self.video.cur_h = if cur_h > 0 { cur_h } else { self.video.height };
                    }

                    if chunk_size == 11 && self.subtype == FlicType::PTF {
                        br.read_skip(1)?;
                    }
                    while br.tell() < chunk_end {
                        let subchunk_size = br.read_u32le()? as usize;
                        let mut subchunk_type = br.read_u16le()?;
                        //println!(" sub {subchunk_type} size {subchunk_size:X} @ {:X}", br.tell() - 6);
                        validate!(subchunk_size >= 6);
                        let mut subchunk_end = br.tell() + (subchunk_size as u64) - 6;
                        if self.aligned {
                            validate!(subchunk_size & 1 == 0);
                        } else if self.subtype == FlicType::AlienVirus {
                            if (chunk_end & 1) == 1 && subchunk_end == chunk_end + 1 {
                                subchunk_end -= 1;
                            }
                            if subchunk_end < chunk_end && (chunk_end - subchunk_end) <= 2 {
                                subchunk_end = chunk_end;
                            }
                        }
                        match self.subtype {
                            FlicType::FLI | FlicType::FLC | FlicType::MagicCarpet |
                            FlicType::Origin | FlicType::PTF | FlicType::CFO |
                            FlicType::AlienVirus | FlicType::Bureau13 => {
                                validate!(subchunk_type < 100);
                            },
                            FlicType::FLH => {
                                validate!(subchunk_type < 100 || subchunk_type == 0x5555 || subchunk_type == 0x5344);
                            }
                        }
                        if self.subtype != FlicType::Origin {
                            validate!(subchunk_end <= chunk_end);
                        } else {
                            validate!(subchunk_end <= (chunk_end + 1) & !1);
                        }

                        // FLH audio
                        if subchunk_type == 0x5344 {
                            validate!((subchunk_size & 1) == 0);
                            self.audio16.resize((subchunk_size - 6) / 2, 0);
                            for el in self.audio16.iter_mut() {
                                *el = br.read_u16le()? as i16;
                            }
                            continue;
                        }

                        // Magic Carpet palette is VGA but misreported as full 256-level one
                        if self.subtype == FlicType::MagicCarpet && subchunk_type == 4 {
                            subchunk_type = 11;
                        }
                        if self.subtype == FlicType::FLI {
                            validate!(subchunk_type != 7);
                        }

                        self.video.handle_subchunk(br.as_mut(), subchunk_type, subchunk_size - 6)?;
                        if self.aligned {
                            if (br.tell() & 1) != 0 {
                                br.read_skip(1)?;
                            }
                            validate!(br.tell() == subchunk_end);
                        } else {
                            validate!(br.tell() <= subchunk_end);
                            if self.subtype != FlicType::Origin {
                                br.seek(SeekFrom::Start(subchunk_end))?;
                            } else {
                                br.seek(SeekFrom::Start(subchunk_end.min(chunk_end)))?;
                            }
                        }
                    }

                    self.frm_no += 1;

                    if let CustomData::B13(ref mut data) = self.custom {
                        if self.video.width == data.orig_w && self.video.height == data.orig_h {
                            return Ok((0, Frame::VideoPal(self.video.frame.clone(), self.video.pal)));
                        }
                        for (dline, sline) in data.vbuf.chunks_exact_mut(data.orig_w)
                                .zip(self.video.frame.chunks_exact(self.video.width))
                                .take(self.video.height) {
                            dline[..self.video.width].copy_from_slice(sline);
                        }
                        return Ok((0, Frame::VideoPal(data.vbuf.clone(), self.video.pal)));
                    }

                    match self.video.depth {
                        1..=8 => {
                            let pal = self.video.pal;
                            let frame = self.video.frame.clone();
                            return Ok((0, Frame::VideoPal(frame, pal)));
                        },
                        15 | 16 => {
                            return Ok((0, Frame::VideoRGB16(self.video.frame16.clone())));
                        },
                        _ => unimplemented!(),
                    }
                },
                0xF1FB => return Err(DecoderError::NotImplemented), // segment table
                0xF1FC => return Err(DecoderError::NotImplemented), // Huffman table
                0xFAF1 if self.subtype == FlicType::CFO => {
                    br.read_skip(chunk_size)?;
                },
                _ => {
                    println!(" skipping unknown chunk {chunk_type:04X} @ {:X}", br.tell() - 6);
                    br.read_skip(chunk_size)?;
                },
            }
        }
        Err(DecoderError::EOF)
    }
}

pub fn open_av(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let size        = br.read_u32le()?;
    let tag         = br.read_u16le()?;
    validate!(tag == 0x3333);
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let depth       = br.read_u16le()?;
    validate!(depth == 8);
    let _flags      = br.read_u16le()?;
    let _speed      = br.read_u32le()?;
                      br.read_u16le()?; // reserved
    let _created    = br.read_u32le()?;
    let creator     = br.read_tag()?;
    if &creator == b"EGI\x00" {
        return Err(DecoderError::NotImplemented);
    }
    let _updated    = br.read_u32le()?;
    let _updater    = br.read_tag()?;
    let _aspect_dx  = br.read_u16le()?;
    let _aspect_dy  = br.read_u16le()?;
    let _ext_flags  = br.read_u16le()?;
    let _keyframes  = br.read_u16le()?;
    let _tot_frames = br.read_u16le()?;
    let _req_mem    = br.read_u32le()?;
    let _max_regs   = br.read_u16le()?;
    let _transp_num = br.read_u16le()?;
                      br.read_skip(24)?;
    let _oframe1    = br.read_u32le()?;
    let _oframe2    = br.read_u32le()?;
                      br.read_skip(40)?;

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes,
        speed: 100,
        fli_end: u64::from(if size > 128 { size } else { std::u32::MAX }),
        tot_size: u64::from(if size > 128 { size } else { std::u32::MAX }),
        subtype: FlicType::AlienVirus,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

pub fn open_b13(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag         = br.read_u16le()?;
    validate!(tag == 0);
                      br.read_u32le()?;
                      br.read_u32le()?;
    let arate       = br.read_u32le()?;
                      br.read_skip(12)?;
    let ssize       = br.read_u32le()?;
    let next_sync = br.tell() + u64::from(ssize);

    let mut abuf = Vec::new();
    let mut abuf_len = 0;
    let nframes;
    let width;
    let height;
    let speed;
    loop {
        let ctype   = br.read_byte()?;
                      br.read_byte()?;
        let size    = br.read_u32le()? as usize;
        match ctype {
            0x02 => {
                abuf_len = size;
                if size > 0 {
                      br.read_extend(&mut abuf, size)?;
                }
            },
            0x03 => {
                validate!(size >= 0x80);
                let _size   = br.read_u32le()?;
                let tag     = br.read_u16le()?;
                validate!(tag == 0xAF11 || tag == 0xAF12);
                nframes     = br.read_u16le()?;
                width       = br.read_u16le()? as usize;
                height      = br.read_u16le()? as usize;
                validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
                let depth   = br.read_u16le()?;
                validate!(depth == 8);
                let _flags  = br.read_u16le()?;
                speed       = br.read_u32le()?;
                              br.read_skip(size - 20)?;
                break;
            },
            0x04 => return Err(DecoderError::InvalidData),
            _ => {
                      br.read_skip(size)?;
            }
        }
    }
    validate!(speed > 0);

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes,
        speed: 100,
        fli_end: 0,
        tot_size: std::u64::MAX,
        subtype: FlicType::Bureau13,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth: 8,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::B13(B13Data{
            vbuf: vec![0; width * height],
            orig_w: width,
            orig_h: height,
            abuf_len,
            next_sync,
            arate,
            abuf,
        }),
    }))
}

fn c13_unpack(br: &mut dyn ByteIO) -> DecoderResult<Vec<u8>> {
    let size = br.read_u32le()? as usize;
    validate!((1..=(1 << 28)).contains(&size));
    let mut dst = Vec::with_capacity(size);
    let mut mask = 0;
    let mut mbits = 0;
    let mut pos = 0;
    let mut win = [0; 0x2000];
    while dst.len() < size {
        if mbits == 0 {
            mask = br.read_byte()?;
            mbits = 8;
        }
        let bit = (mask & 1) != 0;
        mask >>= 1;
        mbits -= 1;
        if bit {
            let b = br.read_byte()?;
            dst.push(b);
            win[pos] = b;
            pos = (pos + 1) & (win.len() - 1);
        } else {
            let offlen = usize::from(br.read_u16le()?);
            let offset = offlen >> 3;
            let len = (offlen & 7) + 2;
            for _ in 0..len {
                win[pos] = win[(pos + win.len() - offset) & (win.len() - 1)];
                dst.push(win[pos]);
                pos = (pos + 1) & (win.len() - 1);
            }
        }
    }
    validate!(dst.len() == size);
    Ok(dst)
}

pub fn open_c13(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let data = c13_unpack(&mut br).map_err(|_| DecoderError::InvalidData)?;
    let mr = VecReader::new_read(data);
    open_fli_core(Box::new(mr))
}

pub fn open_cfo(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag         = br.read_tag()?;
    validate!(&tag == b"CFO\0");
    let _zero       = br.read_u32le()?;
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let mut speed   = br.read_u32le()?;
    validate!(speed < 1000);
    if speed == 0 {
        speed = 10;
    }
    let offset      = br.read_u32le()?;
    validate!(offset >= 22);
    br.seek(SeekFrom::Start(u64::from(offset)))?;

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes,
        speed,
        fli_end: u64::from(std::u32::MAX),
        tot_size: u64::from(std::u32::MAX),
        subtype: FlicType::CFO,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height,
            depth: 8,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

pub fn open_fli(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let br = FileReader::new_read(BufReader::new(file));

    open_fli_core(Box::new(br))
}

fn open_fli_core(mut br: Box<dyn ByteIO>) -> DecoderResult<Box<dyn InputSource>> {
    let size        = br.read_u32le()?;
    let tag         = br.read_u16le()?;
    if !matches!(tag, 0xAF11 | 0xAF12 | 0xAF30 | 0xAF31 | 0xAF44) {
        return Err(DecoderError::InvalidData);
    }
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    if size == 12 { // Magic Carpet FLIC
        return Ok(Box::new(FlicDecoder {
            fr: br,
            frm_no: 0,
            nframes,
            speed: 66,
            fli_end: u64::from(std::u32::MAX),
            tot_size: u64::from(std::u32::MAX),
            subtype: FlicType::MagicCarpet,
            audio16: Vec::new(),
            firstaud: false,
            aligned: true,
            video: VideoData {
                width, height,
                depth: 8,
                cur_w: width,
                cur_h: height,
                pal: [0; 768],
                frame: vec![0; width * height],
                frame16: Vec::new(),
                unp_buf: Vec::new(),
                vdata: Vec::new(),
            },
            custom: CustomData::None,
        }));
    }
    let mut depth   = br.read_u16le()?;
    if depth > 0 && depth <= 24 {
        if !matches!(depth, 8 | 15 | 16 | 24) {
            return Err(DecoderError::NotImplemented);
        }
    } else if tag == 0xAF12 {
        depth = 8;
    } else {
        return Err(DecoderError::InvalidData);
    }
    let _flags      = br.read_u16le()?;
    let speed       = br.read_u32le()?;
                      br.read_u16le()?; // reserved
    let _created    = br.read_u32le()?;
    let creator     = br.read_tag()?;
    if &creator == b"EGI\x00" {
        return Err(DecoderError::NotImplemented);
    }
    let _updated    = br.read_u32le()?;
    let _updater    = br.read_tag()?;
    let _aspect_dx  = br.read_u16le()?;
    let _aspect_dy  = br.read_u16le()?;
    let _ext_flags  = br.read_u16le()?;
    let _keyframes  = br.read_u16le()?;
    let _tot_frames = br.read_u16le()?;
    let _req_mem    = br.read_u32le()?;
    let _max_regs   = br.read_u16le()?;
    let _transp_num = br.read_u16le()?;
                      br.read_skip(24)?;
    let _oframe1    = br.read_u32le()?;
    let _oframe2    = br.read_u32le()?;
                      br.read_skip(40)?;

    Ok(Box::new(FlicDecoder {
        fr: br,
        frm_no: 0,
        nframes,
        speed,
        fli_end: u64::from(if size > 128 { size } else { std::u32::MAX }),
        tot_size: u64::from(if size > 128 { size } else { std::u32::MAX }),
        subtype: if tag == 0xAF11 { FlicType::FLI } else { FlicType::FLC },
        audio16: Vec::new(),
        firstaud: false,
        aligned: true,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

pub fn open_flh(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let size        = br.read_u32le()?;
    let tag         = br.read_u16le()?;
    validate!(tag == 0x1234 || tag == 0xAF43);
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let mut depth       = br.read_u16le()?;
    validate!(depth == 16);
    if tag == 0xAF43 { depth = 15; }
    let _flags      = br.read_u16le()?;
    let mut speed   = br.read_u32le()?;
    validate!(speed < 1000);
    if speed == 0 {
        speed = 10;
    }
                      br.read_u16le()?; // reserved
    let _created    = br.read_u32le()?;
    let _creator    = br.read_tag()?;
    let _updated    = br.read_u32le()?;
    let _updater    = br.read_tag()?;
    let _aspect_dx  = br.read_u16le()?;
    let _aspect_dy  = br.read_u16le()?;
    let _ext_flags  = br.read_u16le()?;
    let _keyframes  = br.read_u16le()?;
    let _tot_frames = br.read_u16le()?;
    let _req_mem    = br.read_u32le()?;
    let _max_regs   = br.read_u16le()?;
    let _transp_num = br.read_u16le()?;
                      br.read_skip(24)?;
    let _oframe1    = br.read_u32le()?;
    let _oframe2    = br.read_u32le()?;
                      br.read_skip(40)?;

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes,
        speed: 1000 / speed,
        fli_end: u64::from(size),
        tot_size: u64::from(size),
        subtype: FlicType::FLH,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: Vec::new(),
            frame16: vec![0; width * height],
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

pub fn open_origin(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag         = br.read_tag()?;
    validate!(&tag == b"FORM");
    let tot_size    = br.read_u32le()?;
    let tag         = br.read_tag()?;
    validate!(&tag == b"INTR" || &tag == b"ENDG");
    let tag         = br.read_tag()?;
    validate!(&tag == b"FLIC");
    br.read_skip(16)?;
    let tag         = br.read_u16le()?;
    validate!(tag == 0xAF11);
    let _nframes    = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let depth       = br.read_u16le()?;
    validate!(depth == 8);
    let _flags      = br.read_u16le()?;
    let speed       = br.read_u32le()?;
                      br.seek(SeekFrom::Start(12))?;

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes: 0,
        speed,
        fli_end: 0,
        tot_size: u64::from(tot_size),
        subtype: FlicType::Origin,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

pub fn open_ptf(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let size        = br.read_u32le()?;
    let tag         = br.read_tag()?;
    validate!(&tag == b".PTF");
    let version     = br.read_u16le()?;
    validate!(version == 0x100);
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width <= 2048 && height <= 1536);
    let depth   = br.read_u16le()?;
    if !matches!(depth, 0 | 8 | 15 | 16 | 24) {
        return Err(DecoderError::NotImplemented);
    }
    validate!((width == 0 && height == 0 && depth == 0) || (width > 0 && height > 0 && depth > 0));
    let _flags      = br.read_u16le()?;
                      br.read_skip(44)?;

    Ok(Box::new(FlicDecoder {
        fr: Box::new(br),
        frm_no: 0,
        nframes,
        speed: 100,
        fli_end: u64::from(size),
        tot_size: u64::from(size),
        subtype: FlicType::PTF,
        audio16: Vec::new(),
        firstaud: true,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
        custom: CustomData::None,
    }))
}

fn sg_unpack_chunk(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let mut br = MemoryReader::new_read(src);
    let mut dict: [usize; 256];
    let mut next = [0; 256];
    let mut stack = [0; 32];
    let mut pos = 0;
    while pos < dst.len() {
        dict = std::array::from_fn(|i| i);
        let mut dpos = 0;
        while dpos < dict.len() {
            let op = usize::from(br.read_byte()?);
            let len = if op > 0x7F {
                    dpos += op - 0x7F;
                    1
                } else { op + 1 };
            if dpos == dict.len() {
                break;
            }
            validate!(dpos + len <= dict.len());
            for (i, (dd, nn)) in dict.iter_mut().zip(next.iter_mut()).enumerate()
                    .skip(dpos).take(len) {
                *dd = usize::from(br.read_byte()?);
                if *dd != i {
                    *nn = usize::from(br.read_byte()?);
                }
            }
            dpos += len;
        }

        let mut nin = usize::from(br.read_u16le()?);
        let mut stkpos = 0;
        loop {
            let idx = if stkpos > 0 {
                    stkpos -= 1;
                    stack[stkpos]
                } else if nin > 0 {
                    nin -= 1;
                    usize::from(br.read_byte()?)
                } else {
                    break;
                };
            if dict[idx] == idx {
                validate!(pos < dst.len());
                dst[pos] = idx as u8;
                pos += 1;
            } else {
                validate!(stkpos + 2 <= stack.len());
                stack[stkpos] = next[idx];
                stack[stkpos + 1] = dict[idx];
                stkpos += 2;
            }
        }
    }
    Ok(())
}

fn sg_unpack(br: &mut dyn ByteIO) -> DecoderResult<Vec<u8>> {
    let tag = br.read_tag()?;
    validate!(&tag == b"PGBP");
    let size = br.read_u32le()? as usize;
    validate!((1..=(1 << 28)).contains(&size));
    let mut dst = vec![0; size];
    let mut schunk = [0; 65536];
    for dchunk in dst.chunks_mut(0x1000) {
        let clen = usize::from(br.read_u16le()?);
        validate!(clen > 0);
        br.read_buf(&mut schunk[..clen])?;
        sg_unpack_chunk(&schunk[..clen], dchunk)?;
    }
    Ok(dst)
}

pub fn open_sg(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let data = sg_unpack(&mut br).map_err(|_| DecoderError::InvalidData)?;
    let mr = VecReader::new_read(data);
    open_fli_core(Box::new(mr))
}
