use std::fs::File;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::imaadpcm::*;

struct DpegDecoder {
    fr:         FileReader<File>,
    version:    u16,
    pal:        [u8; 768],
    fps:        u32,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    vdata:      Vec<u8>,
    adata:      Vec<u8>,
    width:      usize,
    height:     usize,
    stride:     usize,
    arate:      u32,
    channels:   u8,
    audio:      Vec<i16>,
    afrm:       usize,
}

impl DpegDecoder {
    fn blit_intra3(dst: &mut [u8], src: &[u8], width: usize, scale2x: bool, offset: usize) {
        let (stop, sbottom) = src.split_at(src.len() / 2);
        if !scale2x {
            for (dlines, (sline0, sline1)) in dst.chunks_exact_mut(width * 2)
                    .zip(stop.chunks_exact(width / 2).zip(sbottom.chunks(width / 2))) {
                let (dline0, dline1) = dlines.split_at_mut(width);
                let (left, right) = sline0.split_at(sline0.len() / 2);
                for (quad, &p) in dline0.chunks_exact_mut(4).zip(left.iter()) {
                    quad[offset*2]     = p;
                }
                for (quad, &p) in dline1.chunks_exact_mut(4).zip(right.iter()) {
                    quad[offset*2]     = p;
                }
                let (left, right) = sline1.split_at(sline1.len() / 2);
                for (quad, &p) in dline0.chunks_exact_mut(4).zip(left.iter()) {
                    quad[offset*2 + 1] = p;
                }
                for (quad, &p) in dline1.chunks_exact_mut(4).zip(right.iter()) {
                    quad[offset*2 + 1] = p;
                }
            }
        } else if offset == 0 {
            for (dlines, sline) in dst.chunks_exact_mut(width * 2).zip(src.chunks(width / 2)) {
                let (dline0, dline1) = dlines.split_at_mut(width);
                for ((d0, d1), &pix) in dline0.chunks_exact_mut(2)
                        .zip(dline1.chunks_exact_mut(2))
                        .zip(sline.iter()) {
                    d0[0] = pix;
                    d0[1] = pix;
                    d1[0] = pix;
                    d1[1] = pix;
                }
            }
        } else {
            for (dlines, sline) in dst.chunks_exact_mut(width * 2).zip(src.chunks(width / 2 - 1)) {
                let (_dline0, dline1) = dlines.split_at_mut(width);
                for (d1, &pix) in dline1.chunks_exact_mut(2)
                        .zip(sline.iter()) {
                    d1[0] = pix;
                    d1[1] = pix;
                }
            }
        }
    }
    fn decode_inter3(dst: &mut [u8], src: &[u8], width: usize, size: usize) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);

        let mut to_skip = 0;
        let mut pix = None;
        for strip in dst.chunks_exact_mut(width * size) {
            if to_skip >= width {
                to_skip -= width;
                continue;
            }
            let mut x = to_skip;
            to_skip = 0;
            while x < width {
                if to_skip > 0 {
                    to_skip -= size;
                    x += size;
                    continue;
                }
                if let Some(p) = pix {
                    if p != 0 {
                        for line in strip.chunks_exact_mut(width) {
                            for el in line[x..][..size].iter_mut() {
                                *el = p;
                            }
                        }
                        x += size;
                    }
                }
                to_skip = usize::from(br.read_byte()?) * size;
                pix = Some(br.read_byte()?);
            }
        }
        Ok(())
    }
    fn unpack_inter7(dst: &mut [u8], prev: &[u8], width: usize, stride: usize, src: &[u8]) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);

        let end_pos = (prev.len() - stride * 7 - 8) as i32;
        for (tile_y, stripe) in dst.chunks_exact_mut(stride * 8).enumerate() {
            let pos = tile_y * 8 * stride;
            for x in (0..width).step_by(8) {
                let op = br.read_byte()?;
                let offset = if (op >> 6) != 3 {
                        i32::from((((u16::from(op) << 8) | u16::from(br.read_byte()?)) as i16) << 2 >> 2)
                    } else { 0 };
                let copy_pos = ((pos + x) as i32) + offset;
                validate!(copy_pos >= 0 && copy_pos <= end_pos);
                let ref_data = &prev[copy_pos as usize..];

                match op >> 6 {
                    0 => {
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            drow[..8].copy_from_slice(&srow[..8]);
                        }
                    },
                    1 => {
                        let mut run = br.read_byte()?;
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                if run == 0 {
                                    *el = br.read_byte()?;
                                    run = br.read_byte()?;
                                } else {
                                    *el = prev;
                                    run -= 1;
                                }
                            }
                        }
                    },
                    2 => {
                        let mut run = br.read_byte()?;
                        let mut raw = false;
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                match (raw, run == 0) {
                                    (true, true) => {
                                        run = br.read_byte()?;
                                        *el = prev;
                                        raw = false;
                                    },
                                    (true, false) => {
                                        *el = br.read_byte()?;
                                        run -= 1;
                                    },
                                    (false, true) => {
                                        run = br.read_byte()?;
                                        *el = br.read_byte()?;
                                        raw = true;
                                    },
                                    (false, false) => {
                                        *el = prev;
                                        run -= 1;
                                    }
                                }
                            }
                        }
                    },
                    _ => {
                        match op & 0x3F {
                            0x3F => {
                                for drow in stripe[x..].chunks_mut(stride) {
                                    br.read_buf(&mut drow[..8])?;
                                }
                            },
                            0x0F => {
                                for drow in stripe[x..].chunks_mut(stride * 2) {
                                    for xx in (0..8).step_by(2) {
                                        let clr = br.read_byte()?;
                                        drow[xx] = clr;
                                        drow[xx + 1] = clr;
                                        drow[xx + stride] = clr;
                                        drow[xx + stride + 1] = clr;
                                    }
                                }
                            },
                            0x03 => {
                                for drow in stripe[x..].chunks_mut(stride * 4) {
                                    for xx in (0..8).step_by(4) {
                                        let clr = br.read_byte()?;
                                        for quads in drow[xx..].chunks_mut(stride) {
                                            for el in quads[..4].iter_mut() {
                                                *el = clr;
                                            }
                                        }
                                    }
                                }
                            },
                            0x01 => {
                                let offset = i32::from((br.read_u16be()? as i16) << 2 >> 2);
                                let delta  = br.read_u16be()? as i16;
                                let copy_pos = ((pos + x) as i32) + offset;
                                validate!(copy_pos >= 0 && copy_pos <= end_pos);
                                let ref_data = &prev[copy_pos as usize..];
                                for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                                    for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                        *el = (i16::from(prev) + delta).max(0).min(255) as u8;
                                    }
                                }
                            },
                            0x00 => {
                                let clr = br.read_byte()?;
                                for drow in stripe[x..].chunks_mut(stride) {
                                    for el in drow[..8].iter_mut() {
                                        *el = clr;
                                    }
                                }
                            },
                            _ => unimplemented!(),
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn decode_audio7(&mut self) {
        let nsamples = read_u16le(&self.adata).unwrap_or(0) as usize;
        let predictor = (i32::from(self.adata[2]) - 0x80) << 8;
        let step = self.adata[3] as usize;
        let mut ima = IMAState { predictor, step };

        for &b in self.adata[4..][..nsamples / 2].iter() {
            self.audio.push(ima.expand_sample(b & 0xF));
            self.audio.push(ima.expand_sample(b >> 4));
        }
    }
}

impl InputSource for DpegDecoder {
    fn get_num_streams(&self) -> usize { if self.channels > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 100,
                    tb_den: self.fps,
                 }),
            1 if self.channels > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio.len() >= self.afrm {
            let mut audio = vec![0; self.afrm];
            audio.copy_from_slice(&self.audio[..self.afrm]);
            self.audio.drain(..self.afrm);
            return Ok((1, Frame::AudioS16(audio)));
        }

        let br = &mut self.fr;
        match self.version {
            3 => {
                let flags = br.read_byte().map_err(|_| DecoderError::EOF)?;
                let asize = usize::from(br.read_u16le()?);
                self.adata.resize(asize, 0);
                if asize > 0 {
                    br.read_buf(&mut self.adata)?;
                    for &samp in self.adata.iter() {
                        self.audio.push(i16::from(samp ^ 0x80) << 8);
                    }
                }
                if (flags & 1) != 0 {
                    if (flags & 2) != 0 {
                        br.read_vga_pal(&mut self.pal)?;
                    }

                    let vsize = usize::from(br.read_u16le()?);
                    self.vdata.resize(vsize, 0);
                    if vsize > 0 {
                        br.read_buf(&mut self.vdata)?;
                        if (flags & 2) != 0 {
                            Self::blit_intra3(&mut self.frame, &self.vdata, self.width, (flags & 4) != 0, 0);
                        } else {
                            Self::decode_inter3(&mut self.frame, &self.vdata, self.width, 4)
                                .map_err(|_| DecoderError::InvalidData)?;
                        }
                    }

                    let vsize = usize::from(br.read_u16le()?);
                    self.vdata.resize(vsize, 0);
                    if vsize > 0 {
                        br.read_buf(&mut self.vdata)?;
                        if (flags & 2) != 0 {
                            Self::blit_intra3(&mut self.frame, &self.vdata, self.width, (flags & 4) != 0, 1);
                        } else {
                            Self::decode_inter3(&mut self.frame, &self.vdata, self.width, 2)
                                .map_err(|_| DecoderError::InvalidData)?;
                        }
                    }

                    if (flags & 2) == 0 {
                        let vsize = usize::from(br.read_u16le()?);
                        self.vdata.resize(vsize, 0);
                        if vsize > 0 {
                            br.read_buf(&mut self.vdata)?;
                            Self::decode_inter3(&mut self.frame, &self.vdata, self.width, 1)
                                    .map_err(|_| DecoderError::InvalidData)?;
                        }
                    }
                }
                let _smth = br.read_u32le()?;
            },
            7 | 0 => {
                let size  = br.read_u32le()? as usize;
                let flags = br.read_byte()?;
                if size == 0 && flags == 0 {
                    return Err(DecoderError::EOF);
                }

                self.adata.clear();
                if self.channels > 0 {
                    let asize = br.read_u16le()? as usize;
                    validate!(size >= 7 && asize + 7 <= size);

                    self.adata.resize(asize, 0);
                    if !self.adata.is_empty() {
                        br.read_buf(&mut self.adata)?;
                    }
                    self.vdata.resize(size - asize - 7, 0);
                } else {
                    self.vdata.resize(size - 5, 0);
                }
                br.read_buf(&mut self.vdata)?;

                if !self.adata.is_empty() {
                    validate!(self.adata.len() >= 4);
                    self.decode_audio7();
                }
                if (flags & 2) != 0 {
                    validate!(self.vdata.len() > 768);
                    let (paldata, vframe) = self.vdata.split_at(768);
                    for (dst, &p) in self.pal.iter_mut().zip(paldata.iter()) {
                        *dst = (p << 2) | (p >> 4);
                    }
                    if (flags & 4) == 0 {
                        validate!(vframe.len() == self.width * self.height);
                        for (dline, sline) in self.frame.chunks_exact_mut(self.stride)
                                .zip(vframe.chunks_exact(self.width)) {
                            dline[..self.width].copy_from_slice(sline);
                        }
                    } else {
                        validate!(vframe.len() * 2 == self.width * self.height);
                        for (dline, sline) in self.frame.chunks_exact_mut(self.stride)
                                .zip(vframe.chunks_exact(self.width / 2)) {
                            for (dst, &src) in dline.chunks_exact_mut(2).zip(sline.iter()) {
                                dst[0] = src;
                                dst[1] = src;
                            }
                        }
                    }
                } else if (flags & 8) != 0 {
                    std::mem::swap(&mut self.frame, &mut self.pframe);
                    Self::unpack_inter7(&mut self.frame, &self.pframe, self.width, self.stride, &self.vdata)
                        .map_err(|_| DecoderError::InvalidData)?;
                } else {
                    return Err(DecoderError::InvalidData);
                }

                if self.width < 320 {
                    let pal = self.pal;
                    let mut frame = vec![0; self.width * self.height];
                    for (dline, sline) in frame.chunks_exact_mut(self.width)
                            .zip(self.frame.chunks_exact(320)) {
                        dline.copy_from_slice(&sline[..self.width]);
                    }
                    return Ok((0, Frame::VideoPal(frame, pal)));
                }
            },
            _ => return Err(DecoderError::NotImplemented),
        }
        let pal = self.pal;
        let frame = self.frame.to_vec();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    br.read_skip(0x28)?; // first string
    br.read_skip(0x28)?; // second string
    let version = br.read_u16le()?;
    let _version2 = br.read_u16le()?;
    if !matches!(version, 0 | 3 | 7) {
        return Err(DecoderError::NotImplemented);
    }
    br.read_skip(2)?;
    br.read_skip(2)?;
    br.read_skip(2)?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 640);
    validate!(height > 0 && height <= 480);
    br.read_skip(12)?;
    let fps = u32::from(br.read_u16le()?);
    validate!((100..=3000).contains(&fps));
    let arate = u32::from(br.read_u16le()?);
    let channels = br.read_u16le()?;
    validate!(arate > 1000 && arate <= 48000);
    validate!(channels <= 2);
    if channels > 1 { return Err(DecoderError::NotImplemented); }
    let channels = channels as u8;
    br.seek(SeekFrom::Start(0x100))?;

    let stride = if matches!(version, 0 | 7) && width <= 320 { 320 } else { width };

    Ok(Box::new(DpegDecoder {
        fr: br,
        width, height, fps, stride, version,
        arate, channels,
        pal:    [0; 768],
        vdata:  Vec::with_capacity(width * height + 768),
        adata:  Vec::with_capacity(1024),
        frame:  vec![0; stride * height],
        pframe: vec![0; stride * height],
        audio:  Vec::new(),
        afrm:   (arate * 100 / fps).max(1) as usize,
    }))
}
