use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::path::*;

struct CyclemaniaDecoder {
    ffr:        FileReader<BufReader<File>>,
    vfr:        Option<FileReader<BufReader<File>>>,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    cur_frm:    usize,
    nframes:    usize,
    vsizes:     Vec<u32>,
}

impl CyclemaniaDecoder {
    fn unpack_video(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let mut pos = 0;
        while pos < self.frame.len() {
            let op = br.read_byte()?;
            if op == 0 {
                continue;
            }
            match op >> 6 {
                0b00 => {
                    let len = usize::from(op & 0x3F);
                    validate!(pos + len <= self.frame.len());
                    br.read_buf(&mut self.frame[pos..][..len])?;
                    pos += len;
                },
                0b01 => {
                    let len = usize::from(op & 0x3F);
                    let clr = br.read_byte()?;
                    validate!(pos + len <= self.frame.len());
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                _ => {
                    let skip = usize::from(op & 0x7F);
                    validate!(pos + skip <= self.frame.len());
                    pos += skip;
                },
            }
        }

        Ok(())
    }
}

impl InputSource for CyclemaniaDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }

        if self.cur_frm > 0 {
            // first frame is already loaded from .frm, others need to be retrieved from .vid
            if let Some(ref mut vbr) = self.vfr {
                let csize = self.vsizes[self.cur_frm - 1] as usize;
                self.data.resize(csize, 0);
                vbr.read_buf(&mut self.data)?;
            }
        }
        self.cur_frm += 1;

        self.unpack_video().map_err(|_| DecoderError::InvalidData)?;

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let mut bname = name.to_owned();
    match (bname.rfind('.'), bname.rfind(std::path::MAIN_SEPARATOR)) {
        (Some(epos), Some(spos)) => {
            if epos > spos {
                bname.truncate(epos);
            }
        },
        (Some(epos), None) => {
            bname.truncate(epos);
        },
        _ => {}
    }

    //println!("basename: '{bname}'");

    let pal_name = bname.clone() + ".col";
    let file = open_file_igncase(&pal_name).map_err(|_| DecoderError::AuxInputNotFound(pal_name))?;
    let mut br = FileReader::new_read(file);
    let pal_size = br.read_u32le()?;
    validate!(pal_size == 0x308);
    let some_val = br.read_u32le()?;
    validate!(some_val == 0xB123);
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    let hdr_name = bname.clone() + ".hed";
    let file = open_file_igncase(&hdr_name).map_err(|_| DecoderError::AuxInputNotFound(hdr_name))?;
    let mut br = FileReader::new_read(file);
    let nframes = br.read_u32le()? as usize;
    validate!((1..=1024).contains(&nframes));
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    let method = br.read_u32le()?;
    validate!(method < 3);
    let mut vsizes = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        if let Ok(val) = br.read_u32le() {
            vsizes.push(val);
        } else {
            break;
        }
    }

    let frm_name = bname.clone() + ".frm";
    let file = open_file_igncase(&frm_name).map_err(|_| DecoderError::AuxInputNotFound(frm_name))?;
    let mut ffr = FileReader::new_read(BufReader::new(file));
    let size = ffr.read_u32le()? as usize;
    validate!(size < 65536);
    let mut data = vec![0; size];
    ffr.read_buf(&mut data)?;

    let vid_name = bname.clone() + ".vid";
    let vfr = if let Ok(file) = open_file_igncase(&vid_name) {
            Some(FileReader::new_read(BufReader::new(file)))
        } else {
            if !vsizes.is_empty() {
                return Err(DecoderError::AuxInputNotFound(vid_name));
            }
            None
        };

    Ok(Box::new(CyclemaniaDecoder {
        ffr, vfr,
        pal,
        frame: vec![0; width * height],
        data,
        width, height,
        vsizes, nframes,
        cur_frm: 0,
    }))
}
