use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const ARATE: u32 = 11025;

type ChunkHandler = fn(dec: &mut Decoder, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()>;
type ParseList = &'static [(&'static [u8; 4], ChunkHandler)];

const ROOT_CHUNKS: ParseList = &[
    (b"RGBS", Decoder::parse_rgbs),
    (b"SFXB", Decoder::parse_sfxb),
    (b"DATA", Decoder::found_data),
];

const SFXB_CHUNKS: ParseList = &[
    (b"DATA", Decoder::parse_sfxb_data),
];

const FRAME_CHUNKS: ParseList = &[
    (b"BLOK", Decoder::parse_blok),
];

const BLOK_CHUNKS: ParseList = &[
    (b"RATE", Decoder::parse_rate),
    (b"RGBS", Decoder::parse_rgbs),
    (b"SNDE", Decoder::parse_snde),
    (b"LZSS", Decoder::parse_lzss),
    (b"FRAM", Decoder::decode_fram),
    (b"SRLE", Decoder::decode_srle),
    (b"TOIL", Decoder::parse_toil),
];

const VFRAME_CHUNKS: ParseList = &[
    (b"FRAM", Decoder::decode_fram),
    (b"SRLE", Decoder::decode_srle),
];

const MODE_LOOP: u8 = 2;

struct AudioTrack {
    data:       Vec<u8>,
    mode:       u8,
    pos:        usize,
}

impl AudioTrack {
    fn mix(&mut self, dst: &mut [u8]) -> bool {
        if (self.mode & MODE_LOOP) == 0 {
            for el in dst.iter_mut() {
                let ssamp = i16::from(*el) - 0x80;
                let nsamp = i16::from(self.data[self.pos]) - 0x80;
                *el = (ssamp + nsamp + 0x80).max(0).min(0xFF) as u8;
                self.pos += 1;
                if self.pos == self.data.len() {
                    self.pos = 0;
                }
            }
            false
        } else {
            let add_len = (self.data.len() - self.pos).min(dst.len());
            for (el, &add) in dst.iter_mut().zip(self.data[self.pos..].iter()) {
                *el = (i16::from(*el) + i16::from(add) - 0x80).max(0).min(0xFF) as u8;
            }
            self.pos += add_len;
            self.pos == self.data.len()
        }
    }
}

#[derive(Default)]
struct Mixer {
    tracks:     Vec<Option<AudioTrack>>,
}

impl Mixer {
    fn mix(&mut self, dst: &mut [u8]) {
        for track in self.tracks.iter_mut() {
            if let Some(ref mut trk) = track {
                let done = trk.mix(dst);
                if done {
                    *track = None;
                }
            }
        }
    }
    fn add(&mut self, chan: usize, track: AudioTrack) {
        while self.tracks.len() <= chan {
            self.tracks.push(None);
        }
        self.tracks[chan] = Some(track);
    }
    fn stop_all(&mut self) {
        for track in self.tracks.iter_mut() {
            *track = None;
        }
    }
    fn stop_last(&mut self) {
    }
    fn stop(&mut self, id: usize) {
        if id < self.tracks.len() {
            self.tracks[id] = None;
        }
    }
}

struct Decoder {
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    width:      usize,
    height:     usize,
    data_start: u64,
    data_end:   u64,
    sfx:        Vec<Vec<u8>>,
    vend:       u32,
    tick:       u32,
    rate:       u32,
    mixer:      Mixer,
}

impl Decoder {
    fn new(width: usize, height: usize, tick: u32) -> Self {
        Self {
            pal: [0; 768],
            data: Vec::new(),
            udata: Vec::new(),
            frame: vec![0; width * height],
            width, height,
            data_start: 0,
            data_end: 0,
            sfx: Vec::new(),
            vend: 0,
            tick,
            rate: tick,
            mixer: Mixer::default(),
        }
    }
    fn parse_rgbs(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        validate!(size <= self.pal.len() + 8);
        br.read_buf(&mut self.pal[..size - 8])?;
        Ok(())
    }
    fn skip_chunk(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        br.read_skip(size - 8)?;
        Ok(())
    }
    fn found_data(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        self.data_start = br.tell();
        self.data_end = br.tell() + (size as u64) - 8;
        br.read_skip(size - 8)?;
        Ok(())
    }

    fn parse_sfxb(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        validate!(size > 42);
        let tag = br.read_tag()?;
        let wsize = br.read_u32be()? as usize;
        validate!(&tag == b"WRAP" && wsize == size - 8);
        let tag = br.read_tag()?;
        let osize = br.read_u32be()? as usize;
        validate!(&tag == b"OFFS" && osize >= 12 && (osize & 3) == 0);
        let mut poffset = osize as u32 - 9;
        for _ in 0..(osize - 8) / 4 {
            let offset = br.read_u32le()?;
            validate!(offset > poffset + 8 && offset < (size - osize - 8) as u32);
            poffset = offset;
        }
        self.sfx.clear();
        parse_chunks(self, br, size - 8 - osize, SFXB_CHUNKS)
    }
    fn parse_sfxb_data(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        let mut data = vec![0; size - 8];
        br.read_buf(&mut data)?;
        self.sfx.push(data);
        Ok(())
    }

    fn parse_blok(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        self.vend += self.rate;
        parse_chunks(self, br, size, BLOK_CHUNKS)
    }

    fn parse_frame_chunk(&mut self, br: &mut dyn ByteIO) -> DecoderResult<()> {
        while br.tell() < self.data_end {
            let tag = br.read_tag()?;
            let size = br.read_u32be()? as usize;
            validate!(size >= 8);
            let mut found = false;
            for (&ptag, pfunc) in FRAME_CHUNKS.iter() {
                if ptag == tag {
                    (pfunc)(self, br, size)?;
                    found = true;
                    break;
                }
            }
            if !found {
                br.read_skip(size - 8)?;
            }
            if &tag == b"BLOK" {
                return Ok(())
            }
        }
        validate!(br.tell() == self.data_end);
        Err(DecoderError::EOF)
    }
    fn parse_rate(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        validate!(size >= 10);
        let rate = br.read_u16le()?;
        validate!(rate > 0);
        self.vend -= self.rate;
        self.rate = u32::from(rate);
        self.vend += self.rate;
        br.read_skip(size - 10)?;
        Ok(())
    }
    fn parse_snde(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        validate!(size == 18);
        let flags = br.read_u32le()?;
        let num1  = br.read_u16le()? as usize;
        let _num2 = br.read_u16le()? as usize;
        let x8000 = br.read_u16le()?;
        validate!(x8000 == 0x8000);
        validate!((num1 > 0 && num1 <= self.sfx.len()) || num1 == 0xFFFF);
        if num1 != 0xFFFF {
            self.mixer.add(num1 - 1, AudioTrack { mode: flags as u8, pos: 0, data: self.sfx[num1 - 1].clone() });
        } else {
            self.mixer.stop_last();
        }
        Ok(())
    }
    fn parse_toil(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        validate!(size > 10);
        let data_end = br.tell() + (size as u64) - 8;
        let nops = br.read_u16le()? as usize;
        for _ in 0..nops {
            let smth = br.read_byte()?;
            if smth == 0 {
                br.read_u16le()?;
            }
            let mut op = u16::from(br.read_byte()?);
            if op == 0 {
                op = br.read_u16le()?;
            }
            match op {
                1 => self.mixer.stop_all(),
                7 => {
                    let ch_id = br.read_u32le()? as usize;
                    self.mixer.stop(ch_id);
                },
                _ => return Err(DecoderError::NotImplemented),
            }
        }

        validate!(br.tell() == data_end);
        Ok(())
    }
    fn parse_lzss(&mut self, br: &mut dyn ByteIO, lzsize: usize) -> DecoderResult<()> {
        validate!(lzsize > 42);
        let tag = br.read_tag()?;
        let size = br.read_u32be()?;
        validate!(&tag == b"LZHD" && size == 16);
        let x2000 = br.read_u32le()?;
        let csize = br.read_u32le()? as usize;
        validate!(x2000 == 0x2000 && csize > 8);
        let tag = br.read_tag()?;
        let dsize = br.read_u32be()? as usize;
        validate!(&tag == b"DATA" && dsize + 24 == lzsize);
        let lit_offs = br.read_u32le()? as usize;
        let off_offs = br.read_u32le()? as usize;
        self.data.resize(dsize - 16, 0);
        br.read_buf(&mut self.data)?;
        validate!(lit_offs >= 8 && lit_offs < dsize - 8);
        validate!(off_offs >= 8 && off_offs < dsize - 8);
        self.udata.resize(csize, 0);
        lzss_unpack(&mut self.udata, &self.data, &self.data[lit_offs - 8..], &self.data[off_offs - 8..])?;

        let mut cdata = Vec::new();
        std::mem::swap(&mut cdata, &mut self.udata);

        let mut mbr = MemoryReader::new_read(&cdata);
        let ret = parse_chunks(self, &mut mbr, csize + 8, VFRAME_CHUNKS);
        std::mem::swap(&mut cdata, &mut self.udata);
        ret?;

        Ok(())
    }

    fn decode_fram(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        let data_end = br.tell() + (size as u64) - 8;

        let flags = br.read_byte()?;
        let ftype = if (flags & 1) != 0 { u16::from(br.read_byte()?) } else { 0x100 };
        let (left, top, right, bottom) = if (flags & 2) != 0 {
                (br.read_u16le()? as usize, br.read_u16le()? as usize,
                 br.read_u16le()? as usize, br.read_u16le()? as usize)
            } else { (0, 0, self.width - 1, self.height - 1) };
        validate!(right >= left && bottom >= top && right < self.width && bottom < self.height);

        if (flags & 0x80) != 0 {
            validate!(ftype == 0x100);

            for line in self.frame.chunks_exact_mut(self.width).skip(top).take(bottom - top + 1) {
                let dsize = br.read_u16le()?;
                let end = br.tell() + u64::from(dsize);
                let mut x = left;
                while br.tell() < end && x <= right {
                    let op = br.read_byte()?;
                    match op & 3 {
                        1 | 3 => {
                            x += usize::from(op >> 1);
                        },
                        2 => {
                            let len = usize::from(op >> 2) + 1;
                            validate!(x + len <= right + 1);
                            let clr = br.read_byte()?;
                            for el in line[x..][..len].iter_mut() {
                                *el = clr;
                            }
                            x += len;
                        },
                        _ => {
                            let len = usize::from(op >> 2) + 1;
                            validate!(x + len <= right + 1);
                            br.read_buf(&mut line[x..][..len])?;
                            x += len;
                        },
                    }
                }
                validate!(br.tell() == end);
            }
        }

        if (br.tell() & 1) != 0 && br.tell() < data_end {
            br.read_skip(1)?;
        }
        validate!(br.tell() == data_end);

        Ok(())
    }
    fn decode_srle(&mut self, br: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
        let data_end = br.tell() + (size as u64) - 8;

        let left   = br.read_u16le()? as usize;
        let top    = br.read_u16le()? as usize;
        let right  = br.read_u16le()? as usize;
        let bottom = br.read_u16le()? as usize;
        validate!(right >= left && bottom >= top && right < self.width && bottom < self.height);

        let mut remap = [0; 32];
        br.read_buf(&mut remap)?;
        let pic_size = br.read_u32le()? as usize;
        validate!(pic_size <= self.width * self.height);

        let mut pos = 0;
        while br.tell() < data_end && pos < pic_size {
            let op = br.read_byte()?;
            match op & 7 {
                0 | 4 => {
                    let len = usize::from(op >> 2) + 1;
                    pos += len;
                },
                2 => {
                    self.frame[pos] = remap[usize::from(op >> 3)];
                    pos += 1;
                },
                6 => {
                    let len = if op == 6 { usize::from(br.read_byte()?) + 1 } else { usize::from(op >> 3) };
                    let clr = br.read_byte()?;

                    validate!(pos + len <= pic_size);
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                _ => {
                    let len = if op == 1 { usize::from(br.read_u16le()?) + 1 } else { usize::from(op >> 1) };
                    pos += len;
                },
            }
        }

        if (br.tell() & 1) != 0 && br.tell() < data_end {
            br.read_skip(1)?;
        }
        validate!(br.tell() == data_end);

        Ok(())
    }
}

fn lzss_unpack(dst: &mut [u8], flags: &[u8], literals: &[u8], offsets: &[u8]) -> DecoderResult<()> {
    let mut win = [0; 0x1000];
    let mut wpos = 1;

    let mut lits = literals.iter();
    let mut offs = offsets.chunks_exact(2);
    let mut dpos = 0;
    for &flg in flags {
        for i in 0..8 {
            if (flg & (1 << i)) != 0 {
                let lit = lits.next().ok_or(DecoderError::ShortData)?;
                win[wpos] = *lit;
                dst[dpos] = *lit;
                wpos = (wpos + 1) & 0xFFF;
                dpos += 1;
            } else {
                let off = offs.next().ok_or(DecoderError::ShortData)?;
                let offset = read_u16le(off)? as usize;
                let len = (offset >> 12) + 2;
                let mut offset = offset & 0xFFF;
                if offset == 0 {
                    return Ok(());
                }
                for _ in 0..len {
                    win[wpos] = win[offset];
                    dst[dpos] = win[offset];
                    wpos = (wpos + 1) & 0xFFF;
                    offset = (offset + 1) & 0xFFF;
                    dpos += 1;
                }
            }
        }
    }
    Ok(())
}

fn parse_chunks(dec: &mut Decoder, br: &mut dyn ByteIO, tot_size: usize, chunks: ParseList) -> DecoderResult<()> {
    let end = br.tell() + (tot_size as u64) - 8;
    while br.tell() < end {
        let tag = br.read_tag()?;
        let size = br.read_u32be()? as usize;
        validate!(size >= 8);
        let mut found = false;
        for (&ptag, pfunc) in chunks.iter() {
            if ptag == tag {
                (pfunc)(dec, br, size)?;
                found = true;
                break;
            }
        }
        if !found {
            br.read_skip(size - 8)?;
        }
    }
    validate!(br.tell() == end);
    Ok(())
}

struct CupDecoder {
    fr:         FileReader<BufReader<File>>,
    dec:        Decoder,
    ablk_size:  usize,
    time:       u32,
    audio:      bool,
}

impl InputSource for CupDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.dec.width,
                height: self.dec.height,
                bpp:    8,
                tb_num: self.dec.tick,
                tb_den: 1000,
            }),
            1 => StreamInfo::Audio(AudioInfo{
                sample_rate: ARATE,
                channels:    1,
                sample_type: AudioSample::U8,
            }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        loop {
            if self.time < self.dec.vend {
                if !self.audio {
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.dec.frame.clone(), self.dec.pal)));
                } else {
                    self.audio = false;
                    let mut audio = vec![0x80; self.ablk_size];
                    self.dec.mixer.mix(&mut audio);
                    self.time += self.dec.tick;
                    return Ok((1, Frame::AudioU8(audio)));
                }
            }
            self.dec.parse_frame_chunk(&mut *br)?;
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"BEAN");
    let tot_size = br.read_u32be()? as usize;
    validate!(tot_size > 1024);
    let tag = br.read_tag()?;
    validate!(&tag == b"HEAD");
    let head_size = br.read_u32be()? as usize;
    validate!(head_size >= 14);
    let tick = u32::from(br.read_u16le()?);
    validate!(tick > 0);
    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=1024).contains(&width) && (1..=768).contains(&height));
    br.read_skip(head_size - 14)?;

    let mut dec = Decoder::new(width, height, tick);
    parse_chunks(&mut dec, &mut br, tot_size - head_size, ROOT_CHUNKS)?;
    validate!(dec.data_end > 0);
    br.seek(SeekFrom::Start(dec.data_start))?;

    Ok(Box::new(CupDecoder {
        fr: br,
        dec,
        ablk_size: (ARATE * tick / 1000).max(32) as usize,
        time: 0,
        audio: false,
    }))
}
