use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const WIDTH: usize = 320;

struct CRHDecoder {
    fr:         FileReader<BufReader<File>>,
    planes:     [Vec<u8>; 3],
    pplanes:    [Vec<u8>; 3],
    vdata:      Vec<u8>,
    asize:      usize,
    arate:      u32,
    channels:   u8,
    audio:      bool,
    height:     usize,
    fps:        u16,
    frameno:    u16,
    nframes:    u16,
    is_de:      bool,
}

impl CRHDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<Vec<u16>> {
        const BITREV: [u8; 16] = [0x0, 0x8, 0x4, 0xC, 0x2, 0xA, 0x6, 0xE, 0x1, 0x9, 0x5, 0xD, 0x3, 0xB, 0x7, 0xF];
        let mut br = BitReader::new(&self.vdata, BitReaderMode::LE);
        let mut tab1 = [0; 32];
        let mut tab2 = [0; 32];

        let w = br.read(8)? as usize;
        let h = br.read(8)? as usize;
        let tile_w = usize::from(BITREV[w & 0xF] * 16 + BITREV[w >> 4]);
        let tile_h = usize::from(BITREV[h & 0xF] * 16 + BITREV[h >> 4]);
        validate!(tile_w > 0 && tile_h > 0);
        let mut frame = vec![0; WIDTH * self.height];
        for (y, strip) in frame.chunks_exact_mut(WIDTH * tile_h).enumerate() {
            let y = y * tile_h;
            for x in (0..WIDTH).step_by(tile_w) {
                let mode  = br.read(2)?;
                validate!(mode != 3);
                let step1 = BITREV[br.read(4)? as usize] * 4;
                let step2 = BITREV[br.read(4)? as usize] * 4;
                let off1  = BITREV[br.read(4)? as usize] * 2;
                let off2  = BITREV[br.read(4)? as usize] * 2;

                gen_tab(&mut tab1, step1, off1);
                gen_tab(&mut tab2, step2, off2);

                for (j, line) in strip[x..].chunks_mut(WIDTH).enumerate() {
                    for (i, el) in line[..tile_w].iter_mut().enumerate() {
                        let mut src_addr = x * 2 + i * 2 + ((y & !1) + (j & !1)) * WIDTH;
                        if (j & 1) != 0 {
                            src_addr += 1;
                        }
                        let c2 = self.planes[2][src_addr] as i16;
                        let c0 = self.planes[0][src_addr] as i16 + tab2[c2 as usize];
                        let c1 = self.planes[1][src_addr] as i16 + tab1[c2 as usize];
                        let (r, g, b) = match mode {
                                0 => (c0, c1, c2),
                                1 => (c2, c0, c1),
                                2 => (c1, c2, c0),
                                _ => unreachable!(),
                            };
                        let r = u16::from(r.max(0).min(31) as u8);
                        let g = u16::from(g.max(0).min(31) as u8);
                        let b = u16::from(b.max(0).min(31) as u8);
                        *el = (r << 10) | (g << 5) | b;
                    }
                }
            }
        }

        Ok(frame)
    }
}

fn gen_tab(tab: &mut [i16; 32], step: u8, off: u8) {
    let mut sum = 0;
    let off = i16::from(off);
    let step = i16::from(step);
    for el in tab.iter_mut() {
        *el = sum / 32 + off - 16;
        sum += step;
    }
}

fn unpack_plane(dst: &mut [u8], prev: &[u8], src: &[u8], is_de: bool) -> DecoderResult<()> {
    let mut br = BitReader::new(src, BitReaderMode::LE);
    let mut dpos = 0;
    while dpos < dst.len() {
        if !br.read_bool()? {
            let val = br.read(5)? as u8;
            let run = br.read(8)? as usize;
            validate!(dpos + run <= dst.len());
            validate!(run > 0);
            for el in dst[dpos..][..run].iter_mut() {
                *el = val;
            }
            dpos += run;
        } else {
            let use_prev = br.read_bool()?;
            let offs = br.read(if is_de { 16 } else { 17 })? as usize;
            let len = br.read(8)? as usize;
            validate!(offs + len <= prev.len());
            validate!(len > 0);
            if use_prev {
                dst[dpos..][..len].copy_from_slice(&prev[offs..][..len]);
            } else {
                for i in 0..len {
                    dst[dpos + i] = dst[offs + i];
                }
            }
            dpos += len;
        }
    }
    Ok(())
}

impl InputSource for CRHDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: self.height,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frameno >= self.nframes {
            return Err(DecoderError::EOF);
        }

        let br = &mut self.fr;

        if self.audio {
            let channels = usize::from(self.channels);
            let mut audio = Vec::with_capacity((self.asize - channels) * 2);
            let mut pred = [0; 2];
            for samp in pred.iter_mut().take(channels) {
                *samp = br.read_u16le()? as i16;
                audio.push(*samp);
            }
            for _ in (0..self.asize - channels * 2).step_by(channels) {
                for pred in pred[..channels].iter_mut() {
                    let delta = i16::from(br.read_byte()? as i8);
                    if (delta & 1) == 1 {
                        *pred = pred.wrapping_add(delta >> 1 << 4);
                    } else {
                        *pred = pred.wrapping_add(delta >> 1 << 9);
                    }
                    audio.push(*pred);
                }
            }
            self.audio = false;
            return Ok((1, Frame::AudioS16(audio)));
        }

        self.frameno += 1;
        self.audio = true;

        let frame_size = br.read_u32le()? as usize;
        let mut part_size = [0; 3];
        for el in part_size.iter_mut() {
            *el = br.read_u32le()? as usize;
        }
        validate!(frame_size > part_size[0] + part_size[1] + part_size[2] + 16);

        std::mem::swap(&mut self.planes, &mut self.pplanes);
        for ((plane, pplane), &size) in self.planes.iter_mut()
                .zip(self.pplanes.iter()).zip(part_size.iter()) {
            self.vdata.resize(size, 0);
            br.read_buf(&mut self.vdata)?;
            unpack_plane(plane, pplane, &self.vdata, self.is_de).map_err(|_| DecoderError::InvalidData)?;
        }
        let tile_size = br.read_u32le()? as usize;
        self.vdata.resize(tile_size, 0);
        br.read_buf(&mut self.vdata)?;
        validate!(frame_size == part_size[0] + part_size[1] + part_size[2] + tile_size + 16);
        let frame = self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open_darkearth(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, true) }
pub fn open_creatures(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, false) }

fn open(name: &str, darkearth: bool) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let version = br.read_u16le()?;
    validate!(version == 1);
    let height = if darkearth { 200 } else { 240 };

    let arate = br.read_u32le()?;
    let _full_size = br.read_u32le()?;
    br.read_u16le()?;
    br.read_u16le()?;
    br.read_u16le()?;
    let channels = br.read_u16le()?;
    validate!(channels <= 2);
    let channels = channels as u8;
    br.read_u32le()?;
    let _byterate = br.read_u32le()?;
    let _bps = br.read_u16le()?;
    br.read_skip(22)?;

    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let fps = br.read_u16le()?;
    validate!(fps > 0 && fps <= 60);

    let asize = (arate * u32::from(channels) / u32::from(fps) + 2) as usize;

    Ok(Box::new(CRHDecoder {
        fr: br,
        height, fps, nframes,
        frameno: 0,
        arate, channels,
        vdata: Vec::new(),
        planes: [vec![0; WIDTH * height], vec![0; WIDTH * height], vec![0; WIDTH * height]],
        pplanes: [vec![0; WIDTH * height], vec![0; WIDTH * height], vec![0; WIDTH * height]],
        audio: true,
        asize,
        is_de: darkearth,
    }))
}
