use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct ByonDecoder {
    fr:      FileReader<File>,
    width:   usize,
    height:  usize,
    pal:     [u8; 768],
    frame:   Vec<u8>,
    vdata:   Vec<u8>,
}

impl ByonDecoder {
    fn unpack(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let mut skip = 0;
        for rows in self.frame.chunks_exact_mut(self.width * 2) {
            for x in (0..self.width).step_by(2) {
                if skip > 0 {
                    skip -= 1;
                    continue;
                }
                let op = br.read_byte()?;
                match op >> 6 {
                    0 => {
                        let op1 = br.read_byte()?;
                        let op2 = br.read_byte()?;
                        let op3 = br.read_byte()?;
                        validate!((op1 | op2 | op3) < 0x40);
                        rows[x] = op;
                        rows[x + 1] = op1;
                        rows[x + self.width] = op2;
                        rows[x + self.width + 1] = op3;
                    },
                    1 => {
                        rows[x] = op & 0x3F;
                        rows[x + 1] = op & 0x3F;
                        rows[x + self.width] = op & 0x3F;
                        rows[x + self.width + 1] = op & 0x3F;
                    },
                    _ => {
                        skip = usize::from(op & 0x7F);
                    },
                }
            }
        }
        Ok(())
    }
}

impl InputSource for ByonDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        let frm_size = br.read_u16le().map_err(|_| DecoderError::EOF)? as usize;
        self.vdata.resize(frm_size, 0);
        br.read_buf(&mut self.vdata)?;
        self.unpack().map_err(|err| if err != DecoderError::ShortData { err }
                else { DecoderError::InvalidData })?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    let mut magic = [0; 10];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"BYON PIZZA");
    let height  = br.read_u16le()? as usize;
    let width   = br.read_u16le()? as usize;
    validate!((2..=320).contains(&width) && (2..=200).contains(&height));
    validate!((width | height) & 1 == 0);
    br.read_skip(2)?;

    let mut pal = [0; 768];
    for (i, clr) in pal.chunks_exact_mut(3).take(64).enumerate() {
        let val = ((63 - i) * 65 / 16) as u8;
        clr[0] = val;
        clr[1] = val;
        clr[2] = val;
    }

    Ok(Box::new(ByonDecoder {
        fr: br,
        width, height, pal,
        frame: vec![0; width * height],
        vdata: Vec::new(),
    }))
}
