use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::imaadpcm::*;
use std::collections::VecDeque;
use super::super::*;

const AUD_SIZE_8:  usize = 0x27100;
const AUD_SIZE_16: usize = 0x1AEC0;
const SAMPLE_RATE: usize = 15360;
const FPS: u32 = 24;
const AFRM_SIZE: usize = SAMPLE_RATE / (FPS as usize) * 2;

enum State {
    Start,
    FLX,
    FrmDecode,
    ReadAudio,
    BlockEnd,
    End
}

struct OrlandoDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    unp_buf:    Vec<u8>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    cur_w:      usize,
    cur_h:      usize,
    cur_fps:    u16,
    tick:       u16,
    bits:       u8,
    vid_ts:     u32,
    aud_ts:     u32,
    num_anims:  usize,
    cur_anim:   usize,
    state:      State,
    data_end:   u64,
    ima:        [IMAState; 2],
    abuf:       Vec<u8>,
    aud_size:   usize,
    aqueue:     VecDeque<(u32, Frame)>,
    vqueue:     VecDeque<(u32, Frame)>,
}

impl OrlandoDecoder {
    fn unpack_frame(&mut self, frm_type: u16) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let unp_size = if frm_type == 2 {
                br.read_u32le()? as usize
            } else { 0 };

        let mut xpos = 0;
        let mut ypos = 0;
        if frm_type != 0 {
            let offset = br.read_u32le()? as usize;
            validate!(offset <= self.cur_w * self.cur_h);
            xpos = offset % self.cur_w;
            ypos = offset / self.cur_w;
        }

        if frm_type == 2 {
            self.unp_buf.clear();
            self.unp_buf.reserve(unp_size);
            while br.left() > 0 {
                let b = br.read_byte()?;
                if b != 0xFF {
                    self.unp_buf.push(b);
                } else {
                    let val = br.read_byte()?;
                    let count = usize::from(br.read_byte()?);
                    for _ in 0..count {
                        self.unp_buf.push(val);
                    }
                }
            }
            validate!(self.unp_buf.len() + 8 == unp_size);

            br = MemoryReader::new_read(&self.unp_buf);
        }

        while br.left() >= 2 && ypos < self.cur_h {
            let skip = usize::from(br.read_byte()?);
            let mut copy = usize::from(br.read_byte()?);
            // a hack for two or three frames in INTRO.AVX
            if (br.left() as usize) < copy { copy = br.left() as usize; }
            xpos += skip;
            while xpos >= self.cur_w {
                xpos -= self.cur_w;
                ypos += 1;
            }
            for _ in 0..copy {
                validate!(xpos < self.cur_w && ypos < self.cur_h);
                let c = br.read_byte()?;
                if c != 0 {
                    self.frame[xpos + ypos * self.width] = c;
                }
                xpos += 1;
                if xpos >= self.cur_w {
                    xpos -= self.cur_w;
                    ypos += 1;
                }
            }
        }

        Ok(())
    }
}

impl InputSource for OrlandoDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: FPS,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: SAMPLE_RATE as u32,
                    channels:    2,
                    sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = &mut self.fr;

        loop {
            if !self.aqueue.is_empty() && !self.vqueue.is_empty() {
                if self.aqueue[0].0 <= self.vqueue[0].0 {
                    let (_ats, afrm) = self.aqueue.pop_front().unwrap();
                    return Ok((1, afrm));
                } else {
                    let (_vts, vfrm) = self.vqueue.pop_front().unwrap();
                    return Ok((0, vfrm));
                }
            }
            match self.state {
                State::Start => {
                    self.state = State::FLX;
                    if self.bits == 8 {
                        self.abuf.resize(AUD_SIZE_8, 0);
                        br.read_buf(&mut self.abuf)?;
                    } else {
                        for _ in 0..4 {
                            let _val = br.read_u32le()?;
                        }
                        for state in self.ima.iter_mut() {
                            state.predictor = i32::from(br.read_u16le()? as i16);
                            let step = br.read_u16le()? as usize;
                            validate!(step < IMA_MAX_STEP.into());
                            state.step = step;
                        }
                        self.abuf.resize(AUD_SIZE_16 - 0x18, 0);
                        br.read_buf(&mut self.abuf)?;
                    }
                },
                State::FLX => {
                    let size1 = br.read_u32le()?;
                    let _size2 = br.read_u32le()?;
                    if self.data_end == 0 {
                        validate!(size1 > 28);
                        self.data_end = br.tell() - 8 + u64::from(size1);
                    }
                    let tag = br.read_tag()?;
                    validate!(&tag == b"FLX\0");
                    let nframes = br.read_u16le()?;
                    validate!(nframes > 0);
                    let width = br.read_u16le()? as usize;
                    let height = br.read_u16le()? as usize;
                    validate!(width > 0 && width <= self.width);
                    validate!(height > 0 && height <= self.height);
                    self.cur_w = width;
                    self.cur_h = height;
                    let fps = br.read_u16le()?;
                    validate!(fps > 0 && fps < 25);
                    self.cur_fps = fps;
                    br.read_skip(8)?;
                    for el in self.frame.iter_mut() {
                        *el = 0;
                    }
                    self.state = State::FrmDecode;
                    self.tick = 0;
                },
                State::FrmDecode => {
                    let frm_type = br.read_u16le()?;
                    // a check for a new FLX signature appearing suddenly
                    if frm_type == 0x4C46 && br.peek_u16le()? == 0x58 {
                        br.seek(SeekFrom::Current(-10))?;
                        self.state = State::FLX;
                        self.cur_anim += 1;
                        continue;
                    }
                    let size = br.read_u32le()? as usize;
                    validate!(size >= 6);
                    match frm_type {
                        0 | 1 | 2 => {
                            validate!(size >= 6);
                            self.vdata.resize(size - 6, 0);
                            self.state = State::ReadAudio;
                            if !self.vdata.is_empty() {
                                br.read_buf(&mut self.vdata)?;
                                self.unpack_frame(frm_type).map_err(|_| DecoderError::InvalidData)?;
                                br = &mut self.fr;
                            }

                            self.tick += 24;
                            while self.tick >= self.cur_fps {
                                self.vqueue.push_back((self.vid_ts, Frame::VideoPal(self.frame.clone(), self.pal)));
                                self.vid_ts += 1;
                                self.tick -= self.cur_fps;
                            }
                        },
                        4 => {
                            match size {
                                0x206 => {
                                    for entry in self.pal.chunks_exact_mut(3) {
                                        let val = br.read_u16le()?;
                                        let r = (val >> 11) as u8;
                                        let g = ((val >> 5) & 0x3F) as u8;
                                        let b = (val & 0x1F) as u8;
                                        entry[0] = (r << 3) | (r >> 2);
                                        entry[1] = (g << 2) | (g >> 4);
                                        entry[2] = (b << 3) | (b >> 2);
                                    }
                                },
                                0x306 => {
                                    br.read_buf(&mut self.pal)?;
                                },
                                _ => return Err(DecoderError::NotImplemented),
                            }
                        },
                        _ => br.read_skip(size - 6)?,
                    }
                },
                State::ReadAudio => {
                    let asize = br.read_u32le()? as usize;
                    validate!(asize > 4);
                    br.read_extend(&mut self.abuf, asize - 4)?;

                    let mut nblocks = 0;
                    if self.bits == 8 {
                        for block in self.abuf.chunks_exact(AFRM_SIZE) {
                            self.aqueue.push_back((self.aud_ts, Frame::AudioU8(block.to_vec())));
                            self.aud_ts += 1;
                            nblocks += 1;
                        }
                    } else {
                        for block in self.abuf.chunks_exact(AFRM_SIZE) {
                            let audio = decode_ima_adpcm(&mut self.ima, block);
                            self.aqueue.push_back((self.aud_ts, Frame::AudioS16(audio)));
                            self.aud_ts += 1;
                            nblocks += 1;
                        }
                    }
                    self.abuf.drain(..nblocks * AFRM_SIZE);

                    if br.tell() == self.data_end {
                        self.state = State::BlockEnd;
                    } else if let Ok(_val) = br.read_u32le() {
                        self.state = State::FrmDecode;
                    } else {
                        self.state = State::End;
                    }
                },
                State::BlockEnd => {
                    if self.cur_anim + 1 >= self.num_anims {
                        return Err(DecoderError::EOF);
                    }
                    let size1 = br.read_u32le()?;
                    let _size2 = br.read_u32le()?;
                    self.data_end = br.tell() + u64::from(size1) - 8;
                    self.state = State::FrmDecode;
                },
                State::End => {
                    if self.cur_anim >= self.num_anims {
                        if let Some((_ats, afrm)) = self.aqueue.pop_front() {
                            return Ok((1, afrm));
                        }
                        if let Some((_vts, vfrm)) = self.vqueue.pop_front() {
                            return Ok((0, vfrm));
                        }
                        return Err(DecoderError::EOF);
                    }
                },
            }
        }
    }
}

fn decode_ima_adpcm(ima: &mut [IMAState; 2], src: &[u8]) -> Vec<i16> {
    let mut audio = Vec::with_capacity(src.len() * 2);
    for data in src.chunks_exact(8) {
        let mut wl = read_u32le(data).unwrap_or_default();
        let mut wr = read_u32le(&data[4..]).unwrap_or_default();
        for _ in 0..8 {
            audio.push(ima[0].expand_sample((wl & 0xF) as u8));
            audio.push(ima[1].expand_sample((wr & 0xF) as u8));
            wl >>= 4;
            wr >>= 4;
        }
    }
    audio
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let num_anims = br.read_u16le()? as usize;
    validate!(num_anims > 0);
    br.read_byte()?;
    let mut bits = br.read_byte()?;

    if bits == 0 { bits = 8; }
    validate!(bits == 8 || bits == 16);

    let offset = match bits {
            8 => AUD_SIZE_8,
            16 => AUD_SIZE_16,
            _ => unreachable!(),
        };
    br.seek(SeekFrom::Start(offset as u64 + 12))?;
    let tag = br.read_tag()?;
    validate!(&tag == b"FLX\0");
    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && height > 0 && width <= 1024 && height <= 768);

    br.seek(SeekFrom::Start(4))?;

    Ok(Box::new(OrlandoDecoder {
        fr: br,
        width, height, num_anims,
        cur_w: 0,
        cur_h: 0,
        cur_fps: 0,
        cur_anim: 0,
        vdata: Vec::new(),
        unp_buf: Vec::new(),
        pal: [0; 768],
        frame: vec![0; width * height],
        state: State::Start,
        data_end: 0,
        ima: [IMAState::new(), IMAState::new()],
        abuf: Vec::new(),
        bits,
        aud_size: 0,
        aud_ts: 0,
        aqueue: VecDeque::new(),
        vid_ts: 0,
        tick: 0,
        vqueue: VecDeque::new(),
    }))
}
