use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

struct ANDecoder {
    fr:         FileReader<File>,
    pals:       Vec<[u8; 768]>,
    pal_idx:    Vec<usize>,
    frame:      [u8; WIDTH * HEIGHT],
    frm_no:     usize,
    vdata:      Vec<u8>,
}

impl ANDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let mut dpos = 0;

        while br.left() > 0 {
            let op = br.read_byte()?;
            let len = if (op & 0x7F) != 0 { (op & 0x7F) as usize } else { br.read_u16le()? as usize };
            validate!(len > 0 && dpos + len <= self.frame.len());
            if (op & 0x80) != 0 {
                br.read_buf(&mut self.frame[dpos..][..len])?;
                dpos += len;
            } else {
                dpos += len;
            }
        }

        Ok(())
    }
}

impl InputSource for ANDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frm_no >= self.pal_idx.len() { return Err(DecoderError::EOF); }

        let br = &mut self.fr;
        let size = br.read_u32le()?;
        self.vdata.resize(size as usize, 0);
        br.read_buf(&mut self.vdata)?;
        self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;

        let pal = self.pals[self.pal_idx[self.frm_no]];
        let frame = self.frame.to_vec();
        self.frm_no += 1;

        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    let idx_size = br.read_u32le()?;
    validate!(idx_size >= 768 + 3);
    let mut pals = Vec::new();
    let mut pal_idx = Vec::new();
    let idx_end = br.tell() + u64::from(idx_size);
    while br.tell() < idx_end {
        let frm_mode = br.read_byte()?;
        let _smth    = br.read_u16le()?;
        if (frm_mode & 0x20) != 0 {
            br.read_skip(5)?;
        }
        if (frm_mode & 0x40) != 0 {
            br.read_skip(1)?;
        }
        if (frm_mode & 0x80) != 0 {
            br.read_skip(1)?;
        }
        if (frm_mode & 0x10) != 0 {
            let mut pal = [0; 768];
            br.read_vga_pal(&mut pal)?;
            pals.push(pal);
        }
        validate!(!pals.is_empty());
        pal_idx.push(pals.len() - 1);
    }
    validate!(br.tell() == idx_end);

    br.read_skip(0x1400)?;

    Ok(Box::new(ANDecoder {
        fr: br,
        pals, pal_idx,
        vdata:  Vec::with_capacity(WIDTH * HEIGHT),
        frame:  [0; WIDTH * HEIGHT],
        frm_no: 0,
    }))
}
