use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::input::util::lzss::*;
use crate::io::byteio::*;
use super::{GenericEntry, GenericArchive};

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let nfiles = usize::from(fr.read_u16le()?);
    validate!((1..=10000).contains(&nfiles));

    let mut entries = Vec::with_capacity(nfiles);
    let mut raw_name = [0; 13];
    for _ in 0..nfiles {
        fr.read_buf(&mut raw_name)?;
        let size = fr.read_u32le()?;
        let offset = fr.read_u32le()?;
        let mut name = String::with_capacity(12);
        fr.read_byte()?;
        for &c in raw_name.iter() {
            if c == 0 {
                break;
            }
            validate!((0x20..=0x7F).contains(&c));
            name.push(c as char);
        }
        validate!(!name.is_empty());
        entries.push(GenericEntry{ name, size, offset });
    }

    Ok(Box::new(GenericArchive {
        fr, entries,
        entry: 0,
    }))
}

struct STK2Entry {
    name:   String,
    offset: u32,
    size:   u32,
    dsize:  u32,
    compr:  bool,
}

struct STK2Archive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<STK2Entry>,
    entry:      usize,
    no_convert: bool,
    cbuf:       Vec<u8>,
    dbuf:       Vec<u8>,
}

impl ArchiveSource for STK2Archive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone())
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        if !entry.compr || self.no_convert {
            copy_data(&mut self.fr, dst, entry.size as usize)
        } else {
            validate!(entry.size > 4);
            let dsize = self.fr.read_u32le()?;
            validate!(dsize == entry.dsize);
            self.cbuf.resize(entry.size as usize - 4, 0);
            self.dbuf.resize(entry.dsize as usize, 0);
            self.fr.read_buf(&mut self.cbuf)?;
            lzss_unpack(&self.cbuf, &mut self.dbuf).map_err(|_| DecoderError::InvalidData)?;
            dst.write_buf(&self.dbuf)?;
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open_v2(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut tag = [0; 6];
    fr.read_buf(&mut tag)?;
    validate!(&tag[..5] == b"STK2." && (tag[5] == b'0' || tag[5] == b'1'));
    fr.read_skip(14)?;
    fr.read_skip(8)?;
    let toc_offset = fr.read_u32le()?;
    validate!(toc_offset > 0x20);
    fr.seek(SeekFrom::Start(toc_offset.into()))?;
    let nfiles = fr.read_u32le()? as usize;
    validate!((1..=10000).contains(&nfiles));
    let metadata_offset = fr.read_u32le()?;
    validate!(metadata_offset > toc_offset + 8);

    let mut entries = Vec::with_capacity(nfiles);
    fr.seek(SeekFrom::Start(metadata_offset.into()))?;
    for _ in 0..nfiles {
        let name_offset = fr.read_u32le()?;
        validate!(name_offset >= toc_offset + 8 && name_offset < metadata_offset);
        let pos = fr.tell();
        fr.seek(SeekFrom::Start(name_offset.into()))?;
        let mut name = String::with_capacity(12);
        while let Ok(c) = fr.read_byte() {
            if c == 0 { break; }
            validate!(c >= 0x20);
            name.push(c as char);
        }
        validate!(!name.is_empty());
        fr.seek(SeekFrom::Start(pos))?;
        fr.read_skip(14)?;
        fr.read_skip(14)?;
        fr.read_skip(8)?;
        let size = fr.read_u32le()?;
        let dsize = fr.read_u32le()?;
        fr.read_skip(5)?;
        let offset = fr.read_u32le()?;
        validate!(offset >= 0x20 && u64::from(offset) + u64::from(size) <= u64::from(toc_offset));
        let compression = fr.read_u32le()?;
        let compr = compression != 0;
        validate!(compr || size == dsize);
        entries.push(STK2Entry { name, offset, size, dsize, compr });
    }
    Ok(Box::new(STK2Archive {
        fr, entries,
        entry: 0,
        no_convert: false,
        cbuf: Vec::new(),
        dbuf: Vec::new(),
    }))
}
