use std::fs::File;
use std::io::BufReader;
use std::collections::HashMap;
use crate::input::*;
use crate::io::byteio::*;
use crate::io::bitreader::*;

const DICT_SIZE: usize = 4096;
const MAX_BITS:   u8 = 12;
const INVALID_POS: usize = 65536;
const START_BITS: u8 = 8;
const RESET_SYM: usize = 256;
const EOS_SYM: usize = RESET_SYM + 1;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
    nsyms:      usize,
    idx_bits:   u8,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
            idx_bits:   0,
            nsyms:      0,
        }
    }
    fn reset(&mut self, bits: u8) {
        self.nsyms    = (1 << bits) + 2;
        self.dict_pos = self.nsyms;
        self.dict_lim = 1 << (bits + 1);
        self.idx_bits = bits + 1;
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut [u8], pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        validate!(pos + tot_len <= dst.len());

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
        validate!(src.len() >= 2);
        let mut br = BitReader::new(src, BitReaderMode::LE);

        self.reset(START_BITS);

        let mut pos = 0;
        let mut lastidx = INVALID_POS;
        loop {
            let idx         = br.read(self.idx_bits)? as usize;
            if idx == RESET_SYM {
                self.reset(START_BITS);
                lastidx = INVALID_POS;
                continue;
            }
            if idx == EOS_SYM {
                break;
            }
            validate!(idx <= self.dict_pos);
            if idx != self.dict_pos {
                let len = self.decode_idx(dst, pos, idx)?;
                if lastidx != INVALID_POS {
                    self.add(lastidx, dst[pos]);
                }
                pos += len;
            } else {
                validate!(lastidx != INVALID_POS);
                let len = self.decode_idx(dst, pos, lastidx)?;
                let lastsym = dst[pos];
                pos += len;
                validate!(pos < dst.len());
                dst[pos] = lastsym;
                pos += 1;
                self.add(lastidx, lastsym);
            }

            lastidx = idx;
            if self.dict_pos == self.dict_lim && self.idx_bits < MAX_BITS {
                self.dict_lim <<= 1;
                self.idx_bits += 1;
            }
        }
        validate!(pos == dst.len());
        Ok(())
    }
}

struct FullPic {
    pal:    [u8; 768],
    pic:    [u8; 320 * 200],
    state:  u8,
}

struct RLBArchive {
    fr:             FileReader<BufReader<File>>,
    res_entries:    Vec<ResEntry>,
    res_slots:      Vec<Slot>,
    base_id:        u16,
    cur_base:       u32,
    cur_type:       u8,
    data:           Vec<u8>,
    udata:          Vec<u8>,
    lzw:            LZWState,
    pal:            [u8; 768],
    pics:           HashMap<u16, FullPic>,
    next_pic:       Option<u16>,
    no_convert:     bool,
}

struct Slot {
    id:         u16,
    packed:     usize,
    unpacked:   usize,
    flags:      u8,
    offset:     u32,
}

impl Slot {
    fn read(br: &mut dyn ByteIO) -> DecoderResult<Self> {
        let id = br.read_u16le()?;
        let mut sizes = [0; 5];
        br.read_buf(&mut sizes)?;
        let flags = br.read_byte()?;
        let (packed, unpacked) = if (flags & 1) == 0 {
                let part0 = usize::from(read_u16le(&sizes)?);
                let part1 = usize::from(read_u16le(&sizes[2..])?);
                let part2 = usize::from(sizes[4]);
                (part0 + ((part2 & 0xF) << 16), part1 + ((part2 >> 4) << 16))
            } else {
                let size = (read_u32le(&sizes)? | (u32::from(sizes[4]) << 20)) as usize;
                (size, size)
            };
        if (flags & 0x20) == 0 {
            validate!(packed == unpacked);
        }
        let offset = br.read_u32le()?;
        validate!(offset > 0);
        Ok(Slot { id, packed, unpacked, flags, offset })
    }
}

struct ResEntry {
    res_id:     u16,
    res_type:   u8,
    offset:     u32,
}

impl ResEntry {
    fn read(br: &mut dyn ByteIO, off_shift: u8) -> DecoderResult<Self> {
        let res_id = br.read_u16le()?;
        if res_id == 0xFFFF {
            return Err(DecoderError::EOF);
        }
        let w1 = br.read_u16le()?;
        let w2 = br.read_u16le()?;
        let res_type = w1 as u8 & 0x1F;
        let offset = (u32::from(w2) | (u32::from(w1 >> 5) << 16)) << off_shift;
        Ok(Self { res_id, res_type, offset })
    }
}

const EXTENSIONS: [&str; 32] = [
    "lib", "strip", "image", "pal", "vis", "snd", "msg",
    "fnt", "ptr", "bnk", "drv", "prio", "ctrl", "walk", "bm", "save", "seq",
    "rt17", "rt18", "rt19", "rt20", "rt21", "rt22", "rt23", "rt24", "rt25",
    "rt26", "rt27", "rt28", "rt29", "rt30", "rt31"
];

impl RLBArchive {
    fn is_image(res_type: u8, size: usize) -> bool { res_type == 18 && size == 128000 }
    fn is_special(res_type: u8, size: usize) -> bool {
        Self::is_image(res_type, size) || res_type == 3 || (res_type == 14 && size == 16000)
    }
    fn process_palette(&mut self) {
        if self.udata.len() < 9 {
            return;
        }
        let start = usize::from(read_u16le(&self.udata).unwrap_or_default());
        let count = usize::from(read_u16le(&self.udata[2..]).unwrap_or_default());
        if start < 256 {
            let copy_len = (((start + count).min(256) - start) * 3).min(self.udata.len() - 6);
            self.pal[start * 3..][..copy_len].copy_from_slice(&self.udata[6..][..copy_len]);
        }
    }
    fn assemble_pic(&mut self, base_id: u16, slot_id: u16, is_pal: bool) {
        if (is_pal || slot_id < 4) && !self.pics.contains_key(&base_id) {
            self.pics.insert(base_id, FullPic {
                    pal: [0; 768],
                    pic: [0; 320 * 200],
                    state: 0,
                });
        }
        if let Some(pic) = self.pics.get_mut(&base_id) {
            if !is_pal {
                let off = (usize::from(slot_id) / 2) * 160 + (usize::from(slot_id) & 1) * 320 * 100;
                for (dline, sline) in pic.pic[off..].chunks_mut(320)
                        .zip(self.udata.chunks_exact(160)) {
                    dline[..160].copy_from_slice(sline);
                }
                pic.state |= 1 << slot_id;
            } else {
                pic.pal.copy_from_slice(&self.pal);
                pic.state |= 0x10;
            }
        }
    }
    // special type 18 is raw 320x400 image in some games
    fn write_image(&self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        dst.write_buf(b"P6\n640 400\n255\n")?;
        let mut line = [0; 320 * 2 * 3];
        for row in self.udata.chunks_exact(320) {
            for (dst, &b) in line.chunks_exact_mut(6).zip(row.iter()) {
                dst[..3].copy_from_slice(&self.pal[usize::from(b) * 3..][..3]);
                dst[3..6].copy_from_slice(&self.pal[usize::from(b) * 3..][..3]);
            }
            dst.write_buf(&line)?;
        }
        Ok(())
    }
    fn write_pic(dst: &mut dyn ByteIO, pic: &FullPic) -> DecoderResult<()> {
        dst.write_buf(b"P6\n320 200\n255\n")?;
        let mut line = [0; 320 * 3];
        for row in pic.pic.chunks_exact(320) {
            for (dst, &b) in line.chunks_exact_mut(3).zip(row.iter()) {
                dst[..3].copy_from_slice(&pic.pal[usize::from(b) * 3..][..3]);
            }
            dst.write_buf(&line)?;
        }
        Ok(())
    }
}

impl ArchiveSource for RLBArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        for (id, pic) in self.pics.iter() {
            if pic.state == 0x1F {
                self.next_pic = Some(*id);
                return Ok(format!("{id:04}.ppm"));
            }
        }
        while self.res_slots.is_empty() {
            if self.res_entries.is_empty() {
                return Err(DecoderError::EOF);
            } else {
                let res_entry = self.res_entries.remove(0);
                self.base_id  = res_entry.res_id;
                self.cur_base = res_entry.offset;
                self.cur_type = res_entry.res_type;
                self.fr.seek(SeekFrom::Start(res_entry.offset.into()))?;

                let tag = self.fr.read_tag()?;
                validate!(&tag == b"TMI-");
                let rtype = self.fr.read_byte()?;
                validate!(rtype == res_entry.res_type);
                let num_slots = usize::from(self.fr.read_byte()?);
                for _ in 0..num_slots {
                    let slot = Slot::read(&mut self.fr)?;
                    self.res_slots.push(slot);
                }
            }
        }
        let slot = &self.res_slots[0];
        let ext = if !self.no_convert && Self::is_image(self.cur_type, slot.unpacked) {
                "ppm"
            } else { EXTENSIONS[usize::from(self.cur_type)] };
        Ok(format!("{:04}_{:04}.{ext}", self.base_id, slot.id))
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        if let Some(id) = self.next_pic {
            let pic = self.pics.remove(&id).unwrap();
            self.next_pic = None;
            return Self::write_pic(dst, &pic);
        }
        let slot = self.res_slots.remove(0);
        self.fr.seek(SeekFrom::Start((self.cur_base + slot.offset).into()))?;
        let is_special = !self.no_convert && Self::is_special(self.cur_type, slot.unpacked);
        if (slot.flags & 0x20) == 0 {
            if !is_special {
                copy_data(&mut self.fr, dst, slot.packed)
            } else {
                self.udata.resize(slot.unpacked, 0);
                self.fr.read_buf(&mut self.udata)?;
                match self.cur_type {
                    3 => {
                        self.process_palette();
                        self.assemble_pic(self.base_id, slot.id, true);
                    },
                    14 if self.udata.len() == 16000 => {
                        self.assemble_pic(self.base_id, slot.id, false);
                    },
                    18 => return self.write_image(dst),
                    _ => {},
                }
                Ok(dst.write_buf(&self.udata)?)
            }
        } else {
            self.data.resize(slot.packed, 0);
            self.fr.read_buf(&mut self.data)?;
            self.udata.resize(slot.unpacked, 0);
            self.lzw.unpack(&self.data, &mut self.udata).map_err(|_| DecoderError::InvalidData)?;
            if is_special {
                match self.cur_type {
                    3 => {
                        self.process_palette();
                        self.assemble_pic(self.base_id, slot.id, true);
                    },
                    14 if self.udata.len() == 16000 => {
                        self.assemble_pic(self.base_id, slot.id, false);
                    },
                    18 => return self.write_image(dst),
                    _ => {},
                }
            }
            Ok(dst.write_buf(&self.udata)?)
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"TMI-");
    let slot_type = fr.read_byte()?;
    validate!(slot_type == 0 || slot_type == 0x20);
    let nslots = usize::from(fr.read_byte()?);
    validate!(nslots == 1);
    let hdr = Slot::read(&mut fr)?;
    let off_shift = if slot_type == 0 { 0 } else { 4 };
    fr.seek(SeekFrom::Start(hdr.offset.into()))?;

    let mut res_entries = Vec::new();
    loop {
        match ResEntry::read(&mut fr, off_shift) {
            Ok(val) => { res_entries.push(val); },
            Err(DecoderError::EOF) => break,
            Err(err) => return Err(err),
        }
    }

    Ok(Box::new(RLBArchive {
        fr, res_entries,
        base_id: 0,
        cur_base: 0,
        cur_type: 0,
        res_slots: Vec::new(),
        data: Vec::new(),
        udata: Vec::new(),
        lzw: LZWState::new(),
        pal: [0; 768],
        pics: HashMap::new(),
        next_pic: None,
        no_convert: false,
    }))
}
