use crate::input::*;

mod ap;
mod b13gl;
mod bd13;
mod beam;
mod ce_wad;
mod cryo;
mod egg;
mod escalz;
mod gbumps;
mod gjd;
mod gk3;
mod hell;
mod horde;
mod m88;
mod mco;
mod mng;
mod rlb;
mod seventhlev;
mod stk;

use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;

struct GenericEntry {
    name:   String,
    offset: u32,
    size:   u32,
}

struct GenericArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<GenericEntry>,
    entry:      usize,
}

impl ArchiveSource for GenericArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone())
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        copy_data(&mut self.fr, dst, entry.size as usize)
    }
    fn set_no_convert(&mut self) {}
}


/// The list of all compiled archive plugins
pub const ARCHIVE_PLUGINS: &[ArchivePlugin] = &[
    ArchivePlugin { name: "7lev_bin", long_name: "7th Level BIN", open: seventhlev::open },
    ArchivePlugin { name: "access_ap", long_name: "Access Software AP", open: ap::open },
    ArchivePlugin { name: "b13_gl", long_name: "Bureau 13 GL", open: b13gl::open },
    ArchivePlugin { name: "bd13_dat", long_name: "Braid Dead 13 DAT", open: bd13::open },
    ArchivePlugin { name: "beam_gob", long_name: "Beam Software GOB", open: beam::open },
    ArchivePlugin { name: "ce_wad", long_name: "Conquest Earth WAD", open: ce_wad::open },
    ArchivePlugin { name: "cryo_archive", long_name: "Cryo Archive", open: cryo::open },
    ArchivePlugin { name: "egg", long_name: "HotHouse EGG", open: egg::open },
    ArchivePlugin { name: "escal-z", long_name: "Escal compressed", open: escalz::open },
    ArchivePlugin { name: "gjd", long_name: "Trilobyte GJD", open: gjd::open },
    ArchivePlugin { name: "gk3_barn", long_name: "Gabriel Knight 3 Barn", open: gk3::open },
    ArchivePlugin { name: "goosebumps", long_name: "Goosebumps CFS", open: gbumps::open },
    ArchivePlugin { name: "hell-lib", long_name: "Hell: A Cyberpunk Thriller library", open: hell::open },
    ArchivePlugin { name: "horde-vdx", long_name: "The Horde VDX", open: horde::open },
    ArchivePlugin { name: "m88", long_name: "M88 Movie", open: m88::open },
    ArchivePlugin { name: "mco", long_name: "SouthPeak Interactive puzzle archive", open: mco::open },
    ArchivePlugin { name: "mng", long_name: "Hostile Waters MNG", open: mng::open },
    ArchivePlugin { name: "rlb", long_name: "Tsunami Media RLB", open: rlb::open },
    ArchivePlugin { name: "stk", long_name: "Coktel Vision STK", open: stk::open },
    ArchivePlugin { name: "stk2", long_name: "Coktel Vision STK2", open: stk::open_v2 },
];
