use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct Entry {
    res_size:       u32,
    pal:            usize,
    data:           Vec<u8>,
}

struct M88Archive {
    fr:             FileReader<BufReader<File>>,
    res_entries:    [Vec<Entry>; 4],
    cur_type:       usize,
    cur_item:       usize,
    no_convert:     bool,
}

impl ArchiveSource for M88Archive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        const RES_TYPES: [&str; 4] = ["pal0", "wav", "ppm", "pal1"];

        while self.cur_type < self.res_entries.len() {
            if self.cur_item >= self.res_entries[self.cur_type].len() {
                self.cur_type += 1;
                self.cur_item = 0;
            } else if !self.no_convert {
                return Ok(format!("res{:04}.{}", self.cur_item, RES_TYPES[self.cur_type]));
            } else {
                return Ok(format!("res{:04}.type{}", self.cur_item, self.cur_type));
            }
        }
        Err(DecoderError::EOF)
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.res_entries[self.cur_type][self.cur_item];
        self.cur_item += 1;
        let size = entry.data.len().min(entry.res_size as usize);

        match self.cur_type {
            1 if !self.no_convert => {
                validate!(size > 0);
                dst.write_buf(b"RIFF")?;
                dst.write_u32le(size as u32 + 36)?;
                dst.write_buf(b"WAVE")?;

                dst.write_buf(b"fmt ")?;
                dst.write_u32le(16)?;
                dst.write_u16le(0x0001)?; // PCM
                dst.write_u16le(1)?;
                dst.write_u32le(22050)?;
                dst.write_u32le(22050)?; // byterate
                dst.write_u16le(1)?; // block align
                dst.write_u16le(8)?;

                dst.write_buf(b"data")?;
                dst.write_buf(&entry.data[..size])?;
            },
            2 if !self.no_convert  => {
                let pal_rid = if (entry.pal & 1) != 0 { 3 } else { 0 };
                let pal_id = entry.pal >> 1;
                let pal = if self.res_entries[pal_rid].len() > pal_id {
                        &self.res_entries[pal_rid][pal_id].data
                    } else {
                        println!("palette not found!");
                        return Err(DecoderError::InvalidData);
                    };
                let width = usize::from(read_u16le(&entry.data[6..]).unwrap_or_default());
                let height = usize::from(read_u16le(&entry.data[8..]).unwrap_or_default());
                validate!((1..=1024).contains(&width) && (1..=1024).contains(&height));
                validate!(entry.data.len() >= width * height + 0x3C);
                let mut line = vec![0; width * 3];
                dst.write_buf(format!("P6\n{width} {height}\n255\n").as_bytes())?;
                for sline in entry.data[0x3C..].chunks_exact(width).take(height) {
                    for (dst, &pix) in line.chunks_exact_mut(3).zip(sline.iter()) {
                        dst.copy_from_slice(&pal[usize::from(pix) * 4..][..3]);
                        dst.swap(0, 2);
                    }
                    dst.write_buf(&line)?;
                }
            },
            _ => {
                if size > 0 {
                    dst.write_buf(&entry.data[..size])?;
                }
            },
        }
        Ok(())
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    fr.seek(SeekFrom::Start(0x8))?;
    let tag = fr.read_tag()?;
    validate!(&tag == b"M441");

    fr.seek(SeekFrom::Start(0x800))?;

    let mut res_entries = [Vec::new(), Vec::new(), Vec::new(), Vec::new()];
    let mut slot = [0; 0x7F8];
    let mut pal = std::usize::MAX;
    while let Ok(id) = fr.read_u32le() {
        validate!((id & !0xFF00) == 0);
        let res_id = (id >> 8) as usize;
        validate!(res_id < res_entries.len());
        let size = fr.read_u32le()?;
        fr.read_buf(&mut slot)?;

        if size != 0 {
            if res_id == 0 || res_id == 3 {
                pal = res_entries[res_id].len() * 2 + if res_id == 0 { 0 } else { 1 };
            }
            res_entries[res_id].push(Entry{
                    res_size: size,
                    data: Vec::new(),
                    pal,
                });
        }
        validate!(!res_entries[res_id].is_empty());
        let entry = res_entries[res_id].last_mut().unwrap();
        entry.data.extend_from_slice(&slot);
    }

    Ok(Box::new(M88Archive {
        fr, res_entries,
        cur_type: 0,
        cur_item: 0,
        no_convert: false,
    }))
}
