use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;
use super::{GenericEntry, GenericArchive};

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let nfiles = usize::from(fr.read_u16le()?);
    validate!(nfiles > 0);
    let toc_offset = fr.read_u32le()?;

    fr.seek(SeekFrom::Start(toc_offset.into()))?;
    let mut entries: Vec<GenericEntry> = Vec::with_capacity(nfiles);
    let mut raw_name = [0; 8];
    let mut prev_off = 6;
    for _ in 0..nfiles {
        let offset = fr.read_u32le()?;
        validate!(offset >= prev_off && offset <= toc_offset);
        fr.read_buf(&mut raw_name)?;
        let mut name = String::with_capacity(12);
        for &c in raw_name.iter() {
            if c == 0 {
                break;
            }
            validate!((0x20..=0x7F).contains(&c));
            name.push(c as char);
        }
        validate!(!name.is_empty());
        if let Some(last) = entries.last_mut() {
            last.size = offset - last.offset;
        }
        entries.push(GenericEntry{ name, size: 0, offset });
        prev_off = offset;
    }
    if let Some(last) = entries.last_mut() {
        last.size = toc_offset - last.offset;
    }

    Ok(Box::new(GenericArchive {
        fr, entries,
        entry: 0,
    }))
}
