use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct GJDArchive {
    fr:         FileReader<BufReader<File>>,
    count:      usize,
}

impl ArchiveSource for GJDArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        while let Ok(0xFF) = self.fr.peek_byte() {
            self.fr.read_byte()?;
        }
        if let Ok(tag) = self.fr.peek_tag() {
            let ext = match &tag {
                    &[0x84, 0x10, 0xFF, 0xFF] | &[0x84, 0x10, 0x00, 0x00] => ".roq",
                    &[0x67, 0x92, _, _] => ".vdx",
                    b"RIFF" | &[0xFF, b'R', b'I', b'F'] => ".wav",
                    _ => return Err(DecoderError::NotImplemented),
                };
            Ok(format!("{:06}{ext}", self.count))
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let tag = self.fr.peek_tag()?;
        match &tag {
            &[0x67, 0x92, _, _] => {
                let tag = self.fr.read_u16be()?;
                //validate!(tag == 0x6792);
                dst.write_u16be(tag)?;
                let mut buf = [0; 6];
                self.fr.read_buf(&mut buf)?;
                dst.write_buf(&buf)?;
                loop {
                    let ctype = self.fr.read_byte()?;
                    dst.write_byte(ctype)?;
                    if ctype == 0xFF {
                        break;
                    }
                    let smth  = self.fr.read_byte()?;
                    dst.write_byte(smth)?;
                    let size = self.fr.read_u32le()?;
                    dst.write_u32le(size)?;
                    let metadata = self.fr.read_u16be()?;
                    dst.write_u16be(metadata)?;
                    copy_data(&mut self.fr, &mut *dst, size as usize)?;
                }
            },
            &[0x84, 0x10, 0xFF, 0xFF] | &[0x84, 0x10, 0x00, 0x00] => {
                let tag = self.fr.read_u16le()?;
                dst.write_u16le(tag)?;
                let size = self.fr.read_u32le()?;
                dst.write_u32le(size)?;
                let fps = self.fr.read_u16le()?;
                dst.write_u16le(fps)?;
                while let Ok(tag) = self.fr.peek_u16le() {
                    if tag == 0x1084 {
                        break;
                    }
                    let tag = self.fr.read_u16le()?;
                    dst.write_u16le(tag)?;
                    let size = self.fr.read_u32le()?;
                    dst.write_u32le(size)?;
                    let args = self.fr.read_u16le()?;
                    dst.write_u16le(args)?;
                    copy_data(&mut self.fr, &mut *dst, size as usize)?;
                }
            },
            b"RIFF" => {
                let tag = self.fr.read_tag()?;
                dst.write_buf(&tag)?;
                let size = self.fr.read_u32le()?;
                dst.write_u32le(size)?;
                copy_data(&mut self.fr, &mut *dst, size as usize)?;
            },
            _ => unreachable!()
        }
        self.count += 1;
        Ok(())
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.peek_tag()?;
    if !matches!(&tag,
            &[0x84, 0x10, 0xFF, 0xFF] | &[0x84, 0x10, 0x00, 0x00] |
            &[0x67, 0x92, _, _] |
            b"RIFF") {
        return Err(DecoderError::InvalidData);
    }
    Ok(Box::new(GJDArchive {
        fr,
        count: 0,
    }))
}
