use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct GoosebumpsArchive {
    fr:         FileReader<BufReader<File>>,
    start:      u32,
    end:        u32,
    is_gvd:     bool,
    no_convert: bool,
}

fn patch_size(bw: &mut dyn ByteIO, pos: u64) -> DecoderResult<()> {
    let size = bw.tell() - pos;
    bw.seek(SeekFrom::Current(-((size + 4) as i64)))?;
    bw.write_u32le(size as u32)?;
    bw.seek(SeekFrom::End(0))?;
    Ok(())
}

fn gvd_to_avi(src: &mut dyn ByteIO, dst: &mut dyn ByteIO, size: usize) -> DecoderResult<()> {
    let start_pos = src.tell();
    let minus_one = src.read_u32le()?;
    validate!(minus_one == 0xFFFFFFFF);
    src.read_u32le()?;
    let nframes = src.read_u32le()? as usize;
    validate!((1..=10000).contains(&nframes));
    src.read_u32le()?;
    src.read_u32le()?;

    let mut offsets = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let off = src.read_u32le()?;
        offsets.push(off & 0x7FFFFFFF);
    }
    for range in offsets.windows(2) {
        validate!(range[0] < range[1]);
    }
    validate!((offsets[offsets.len() - 1] as usize) < size);

    let hdr_start = src.tell() - start_pos;
    validate!(u64::from(offsets[0]) > hdr_start);
    let hdr_size = (u64::from(offsets[0]) - hdr_start) as usize;
    validate!(hdr_size <= 65536);
    let mut hdr = vec![0; hdr_size];
    src.read_buf(&mut hdr)?;

    // patch compression method - 1 looks like a custom codec, not RLE
    if hdr[16..20] == [1, 0, 0, 0] {
        hdr[16..20].copy_from_slice(b"RRAG");
    }

    let has_audio = hdr_size > 0x42C && hdr[0x42C] == 1;

    dst.write_buf(b"RIFF")?;
    dst.write_u32le(0)?;
    dst.write_buf(b"AVI LIST")?;
    dst.write_u32le(0)?;
    dst.write_buf(b"hdrlavih")?;
    dst.write_u32le(56)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(if has_audio { 2 } else { 1 })?;
    dst.write_u32le(0)?;
    dst.write_buf(&hdr[4..][..4])?;
    dst.write_buf(&hdr[8..][..4])?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0)?;
    dst.write_buf(&[0; 56])?;
    dst.write_buf(b"LIST")?;
    dst.write_u32le(0)?;
    let strl_pos = dst.tell();
    dst.write_buf(b"strlstrh")?;
    dst.write_u32le(56)?;
    dst.write_buf(b"vids")?;
    dst.write_buf(b"MSVC")?;
    dst.write_u32le(0)?;
    dst.write_u16le(0)?;
    dst.write_u16le(0)?;
    dst.write_u32le(0)?;
    dst.write_u32le(1)?; // scale
    dst.write_u32le(15)?; // rate
    dst.write_u32le(0)?;
    dst.write_u32le(nframes as u32)?;
    dst.write_u32le(0)?;
    dst.write_u32le(0xFFFFFFFF)?;
    dst.write_u32le(0)?;
    dst.write_u16le(0)?;
    dst.write_u16le(0)?;
    dst.write_u16le(0)?;
    dst.write_u16le(0)?;
    dst.write_buf(b"strf")?;
    dst.write_u32le(hdr_size.min(0x42C) as u32)?;
    dst.write_buf(&hdr[..hdr.len().min(0x42C)])?;
    patch_size(&mut *dst, strl_pos)?;
    if has_audio {
        dst.write_buf(b"LIST")?;
        dst.write_u32le(0)?;
        let strl_pos = dst.tell();
        dst.write_buf(b"strlstrh")?;
        dst.write_u32le(56)?;
        dst.write_buf(b"auds")?;
        dst.write_u32le(0)?;
        dst.write_u32le(0)?;
        dst.write_u16le(0)?;
        dst.write_u16le(0)?;
        dst.write_u32le(0)?;
        dst.write_u32le(1)?; // scale
        dst.write_buf(&hdr[0x430..][..4])?; // rate
        dst.write_u32le(0)?;
        dst.write_u32le(0)?;
        dst.write_u32le(0)?;
        dst.write_u32le(0)?;
        dst.write_u32le(0)?;
        dst.write_u16le(0)?;
        dst.write_u16le(0)?;
        dst.write_u16le(0)?;
        dst.write_u16le(0)?;
        dst.write_buf(b"strf")?;
        dst.write_u32le(16)?;
        dst.write_buf(&hdr[0x42C..][..16])?;
        patch_size(&mut *dst, strl_pos)?;
    }
    patch_size(&mut *dst, 20)?;
    dst.write_buf(b"LIST")?;

    dst.write_u32le(0)?;
    let movi_pos = dst.tell();
    dst.write_buf(b"movi")?;
    for &off in offsets.iter() {
        src.seek(SeekFrom::Start(start_pos + u64::from(off)))?;
        let asize = src.read_u32le()? as usize;
        let vsize = src.read_u32le()? as usize;
        let _smth = src.read_u32le()? as usize;
        if !has_audio {
            src.read_skip(asize)?;
        } else {
            dst.write_buf(b"01wb")?;
            dst.write_u32le(asize as u32)?;
            copy_data(&mut *src, &mut *dst, asize)?;
            if (asize & 1) != 0 {
                dst.write_byte(0)?;
            }
        }
        dst.write_buf(b"00dc")?;
        dst.write_u32le(vsize as u32)?;
        copy_data(&mut *src, &mut *dst, vsize)?;
        if (vsize & 1) != 0 {
            dst.write_byte(0)?;
        }
    }

    patch_size(&mut *dst, movi_pos)?;
    patch_size(&mut *dst, 8)?;

    Ok(())
}

impl ArchiveSource for GoosebumpsArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        let hdr_size = self.fr.read_u32le().map_err(|_| DecoderError::EOF)?;
        let hdr_end = self.fr.tell() + u64::from(hdr_size);
        let start = self.fr.read_u32le()?;
        let end = self.fr.read_u32le()?;
        validate!(u64::from(start) >= hdr_end && start <= end);
        self.fr.read_skip(16)?;
        let mut name = String::new();
        while self.fr.tell() < hdr_end {
            let c = self.fr.read_byte()?;
            if c == 0 {
                break;
            }
            validate!((0x20..=0x7F).contains(&c));
            name.push(c as char);
        }
        self.fr.seek(SeekFrom::Start(start.into()))?;
        self.start = start;
        self.end = end;
        self.is_gvd = !self.no_convert && (name.ends_with(".gvd") || name.ends_with(".GVD"));
        if self.is_gvd {
            name += ".avi";
        }
        Ok(name)
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        self.fr.seek(SeekFrom::Start(self.start.into()))?;
        if !self.is_gvd {
            copy_data(&mut self.fr, dst, (self.end - self.start) as usize)
        } else {
            gvd_to_avi(&mut self.fr, dst, (self.end - self.start) as usize)?;
            self.fr.seek(SeekFrom::Start(self.end.into()))?;
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"FSH2");
    fr.seek(SeekFrom::Start(0x2D))?;
    let offset = fr.read_u32le()?;
    validate!(offset > 0x30);
    fr.seek(SeekFrom::Start(offset.into()))?;

    Ok(Box::new(GoosebumpsArchive {
        fr,
        start: 0,
        end: 0,
        is_gvd: false,
        no_convert: false,
    }))
}
