use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::io::codebook::*;

struct EscalFile {
    fr:             FileReader<BufReader<File>>,
    name:           String,
    packed_size:    usize,
    unpacked_size:  usize,
    packed:         bool,
}

enum CB<T:Copy> {
    Code(Codebook<T>),
    Single(T),
}

trait FromIndex {
    fn from_usize(val: usize) -> Self;
}

impl FromIndex for u8 {
    fn from_usize(val: usize) -> u8 { val as u8 }
}
impl FromIndex for u16 {
    fn from_usize(val: usize) -> u16 { val as u16 }
}

impl<T> CB<T> where T: Copy + FromIndex + 'static {
    fn generate(lengths: &[u8]) -> DecoderResult<Self> {
        let mut codes = [0u32; 512];

        let mut bits = [0; 17];
        let mut pfx = [0; 18];

        for len in lengths.iter() {
            let len = *len as usize;
            if len >= bits.len() {
                return Err(DecoderError::InvalidData);
            }
            bits[len] += 1;
        }
        bits[0] = 0;
        let mut code = 0;
        for i in 0..bits.len() {
            code = (code + bits[i]) << 1;
            pfx[i + 1] = code;
        }

        for (len, code) in lengths.iter().zip(codes.iter_mut()) {
            let len = *len as usize;
            if len != 0 {
                //let bits = len as u8;
                *code = pfx[len];//reverse_bits(pfx[len], bits)
                pfx[len] += 1;
            } else {
                *code = 0;
            }
        }

        let mut cbr = TableCodebookDescReader::new(&codes, lengths, |idx| T::from_usize(idx));
        let cb = Codebook::new(&mut cbr, CodebookMode::MSB).map_err(|_| DecoderError::InvalidData)?;
        Ok(Self::Code(cb))
    }
    fn read(&self, br: &mut BitReader) -> DecoderResult<T> {
        match self {
            CB::Code(ref cb) => {
                Ok(br.read_cb(cb)?)
            },
            CB::Single(ref sym) => Ok(*sym),
        }
    }
}

fn read_len_code_desc(br: &mut BitReader, len_bits: u8, max_codes: usize, gap: bool) -> DecoderResult<CB<u8>> {
    let ncodes = br.read(len_bits)? as usize;
    if ncodes > 0 {
        validate!(ncodes <= max_codes);
        let mut cw_len = [0u8; 32];

        let mut i = 0;
        while i < ncodes {
            cw_len[i] = br.read(3)? as u8;
            if cw_len[i] == 7 {
                while cw_len[i] < 16 && br.read_bool()? {
                    cw_len[i] += 1;
                }
            }
            i += 1;
            if gap && (i == 3) {
                i += br.read(2)? as usize;
            }
        }
        CB::generate(&cw_len[..ncodes])
    } else {
        let sym = br.read(len_bits)? as u8;
        Ok(CB::Single(sym))
    }
}

fn read_lit_codes(br: &mut BitReader, lengths_cb: &CB<u8>) -> DecoderResult<CB<u16>> {
    let ncodes = br.read(9)? as usize;
    if ncodes > 0 {
        let mut i = 0;
        let mut lengths = [0; 512];
        while i < ncodes {
            let nbits = lengths_cb.read(br)?;
            match nbits {
                0 => i += 1,
                1 => i += (br.read(4)? + 3) as usize,
                2 => i += (br.read(9)? + 20) as usize,
                _ => {
                    lengths[i] = nbits - 2;
                    i += 1;
                },
            };
        }
        CB::generate(&lengths[..ncodes])
    } else {
        let code = br.read(9)? as u16;
        Ok(CB::Single(code))
    }
}

fn unpack(src: &[u8], dst: &mut dyn ByteIO, mut dst_size: usize) -> DecoderResult<()> {
    let mut br = BitReader::new(src, BitReaderMode::BE);
    let mut window = [0; 0x2000];
    let mut pos = 0;
    let mut nsyms = 0;

    let mut lit_cb = None;
    let mut off_cb = None;
    while dst_size > 0 {
        if nsyms == 0 {
            nsyms = br.read(16)?;
            validate!(nsyms > 0);
            let lit_lens = read_len_code_desc(&mut br, 5, 19, true)?;
            lit_cb = Some(read_lit_codes(&mut br, &lit_lens)?);
            off_cb = Some(read_len_code_desc(&mut br, 4, 14, false)?);
        }
        nsyms -= 1;
        let sym = if let Some(ref lcb) = lit_cb { lcb.read(&mut br)? } else { unreachable!() };
        if sym < 0x100 {
            window[pos] = sym as u8;
            dst.write_byte(window[pos])?;
            pos = (pos + 1) & 0x1FFF;
            dst_size -= 1;
        } else {
            let length = (sym - 0x100 + 3) as usize;
            let offset_len = if let Some(ref ocb) = off_cb { ocb.read(&mut br)? } else { 0 };
            let offset = match offset_len {
                    0 => 0,
                    1 => 1,
                    _ => ((1 << (offset_len - 1)) | br.read(offset_len - 1)?) as usize
                };
            validate!(dst_size >= length);
            for _ in 0..length {
                window[pos] = window[(pos + 0x2000 - offset - 1) & 0x1FFF];
                dst.write_byte(window[pos])?;
                pos = (pos + 1) & 0x1FFF;
                dst_size -= 1;
            }
        }
    }

    Ok(())
}

impl ArchiveSource for EscalFile {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.name.is_empty() {
            return Err(DecoderError::EOF);
        }
        let mut name = String::new();
        std::mem::swap(&mut name, &mut self.name);
        Ok(name)
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        if self.packed {
            let mut src = vec![0; self.packed_size];
            self.fr.read_buf(&mut src)?;
            unpack(&src, dst, self.unpacked_size).map_err(|_| DecoderError::InvalidData)?;
            Ok(())
        } else {
            copy_data(&mut self.fr, dst, self.packed_size)
        }
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let hdr_size = usize::from(fr.read_byte()?);
    validate!(hdr_size >= 0x16);
    fr.read_byte()?;
    let mut magic = [0; 5];
    fr.read_buf(&mut magic)?;
    let mode = magic[3];
    magic[3] = b'x';
    validate!(&magic == b"xxxxx");
    validate!(matches!(mode, b'0' | b'4' | b'5' | b'x'));

    let packed_size = fr.read_u32le()? as usize;
    let unpacked_size = fr.read_u32le()? as usize;
    validate!((1..=10000000).contains(&packed_size));
    validate!((1..=10000000).contains(&unpacked_size));
    fr.read_skip(6)?;
    let name_len = usize::from(fr.read_byte()?);
    validate!((1..=12).contains(&name_len));
    validate!(name_len + 0x14 <= hdr_size);
    let mut name = String::with_capacity(name_len);
    for _ in 0..name_len {
        let c = fr.read_byte()?;
        validate!((0x20..=0x7F).contains(&c) && c != b':' && c != b'\\');
        name.push(c as char);
    }
    fr.read_skip(hdr_size - 0x14 - name_len)?;

    Ok(Box::new(EscalFile {
        fr,
        name, packed_size, unpacked_size,
        packed: mode != b'0',
    }))
}
