use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;
use super::{GenericEntry, GenericArchive};

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let nfiles = usize::from(fr.read_u16le()?);
    validate!((1..=10000).contains(&nfiles));
    let cat_offset = fr.read_u32le()?;
    validate!(cat_offset > 6);
    fr.seek(SeekFrom::Start(cat_offset.into()))?;

    let mut entries = Vec::with_capacity(nfiles);
    let mut raw_name = [0; 8];
    let mut last_offset = 5;
    for _ in 0..nfiles {
        let offset = fr.read_u32le()?;
        validate!(offset > last_offset && offset < cat_offset);
        last_offset = offset;
        fr.read_buf(&mut raw_name)?;
        let mut name = String::with_capacity(8);
        for &c in raw_name.iter() {
            if c == 0 {
                break;
            }
            validate!((0x20..=0x7F).contains(&c));
            name.push(c as char);
        }
        validate!(!name.is_empty());
        entries.push(GenericEntry{ name, size: 0, offset });
    }
    for i in 0..nfiles - 1 {
        entries[i].size = entries[i + 1].offset - entries[i].offset;
    }
    let last = nfiles - 1;
    entries[last].size = cat_offset - entries[last].offset;

    Ok(Box::new(GenericArchive {
        fr, entries,
        entry: 0,
    }))
}
